import React, {PureComponent, UIEventHandler} from 'react';
import _noop from 'lodash/noop';
import _throttle from 'lodash/throttle';
import {subscribe, unsubscribe} from 'pubsub-js';
import {InView} from 'react-intersection-observer';

import {CALENDAR_EVENTS} from 'components/Calendar/constants/constants';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    ECalendarType,
    ICalendarMonth,
    TWeekDayObject,
    ECalendarScrollSource,
} from 'components/Calendar/types';
import {TCalendarPrices} from 'types/common/calendarPrice/ICalendarPrice';

import {canMoveMonthLabel} from 'components/Calendar/utilities/canMoveMonthLabel';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModDesktop} from 'utilities/stylesUtils';

import CalendarMonth from '../../components/CalendarMonth/CalendarMonth';
import CalendarWeekdays from '../../components/CalendarWeekdays/CalendarWeekdays';
import CalendarScrollController from '../../components/CalendarScrollController/CalendarScrollController';

import cx from './CalendarMonthsGrid.scss';

const SCROLL_THROTTLE_TIME = 30;
const MIN_MONTH_TO_RENDER = 3;
const MONTH_TITLE_HEIGHT = 32;

// Соответствует размерам заданным в CalendarDay.scss
enum ECalendarDayHeight {
    MOBILE = 48,
    DESKTOP = 52,
}

interface ICalendarMonthsGridProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    dayClassName: string;
    weekdaysClassName: string;
    months: ICalendarMonth[];
    weekdays: TWeekDayObject[];
    lastScrollSource: string | null;
    scrollPartPosition: number;
    onChangeScrollPosition: Function;
    nowDate: Date;
    minDate: Date;
    maxDate: Date;
    startDate: Date;
    endDate: Date;
    hoveredDate?: Date;
    isRangeSelected: boolean;
    canScrollIntoSelectedDate: boolean;
    calendarType: ECalendarType;
    prices?: TCalendarPrices;
    onDayClick: Function;
    onDayMouseLeave: Function;
    onDayMouseEnter: Function;

    onTouchStart?: (event: TouchEvent) => void;
    onTouchMove?: (event: TouchEvent) => void;
    onTouchEnd?: (event: TouchEvent) => void;
}

interface ICalendarMonthsGridState {
    maxScrollTop: number;
    withoutScrollAnimation: boolean;
}

class CalendarMonthsGrid extends PureComponent<
    ICalendarMonthsGridProps,
    ICalendarMonthsGridState
> {
    private monthsListNode: HTMLDivElement | null = null;
    private scrolledDayNode: HTMLDivElement | null = null;
    private activeDayRef: HTMLDivElement | null = null;

    static readonly defaultProps = {
        className: '',
        dayClassName: '',
        weekdaysClassName: '',
        months: [],
        weekdays: [],
        lastScrollSource: '',
        scrollPartPosition: 0,
        onChangeScrollPosition: _noop,
        // TODO: Не сходится с типом в ICalendarMonthsGridProps
        startDate: null,
        // TODO: Не сходится с типом в ICalendarMonthsGridProps
        endDate: null,
        calendarType: ECalendarType.StartDate,
        headerNode: null,
        canScrollIntoSelectedDate: true,
        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
    };

    constructor(props: ICalendarMonthsGridProps) {
        super(props);

        this.state = {
            maxScrollTop: 0,
            withoutScrollAnimation: true,
        };

        this.flushScrollPosition = _throttle(
            this.flushScrollPosition,
            SCROLL_THROTTLE_TIME,
        );
    }

    componentDidMount(): void {
        this.subscribe();
        this.scrollIntoScrolledDay();
        this.calculateMonthsListMaxScrollTop();

        this.addTouchEventListeners();
    }

    componentDidUpdate(prevProps: ICalendarMonthsGridProps): void {
        this.checkScrollAnimateStatus(prevProps);
        this.calculateMonthsListMaxScrollTop();
    }

    componentWillUnmount(): void {
        this.unsubscribe();
        this.removeTouchEventListeners();
    }

    private subscribe(): void {
        subscribe(
            CALENDAR_EVENTS.SCROLL_TO_ACTIVE_DATE,
            this.scrollToActiveDate,
        );
    }

    private unsubscribe(): void {
        unsubscribe(CALENDAR_EVENTS.SCROLL_TO_ACTIVE_DATE);
    }

    private addTouchEventListeners(): void {
        const ref = this.getMonthsListRef();

        if (!ref) {
            return;
        }

        const {onTouchStart, onTouchEnd, onTouchMove} = this.props;

        // Добавляем через eventListeners, вместо системы событий реакта,
        // Так как нет возможности указать {passive: false}
        if (onTouchStart) {
            ref.addEventListener('touchstart', onTouchStart);
        }

        if (onTouchMove) {
            ref.addEventListener('touchmove', onTouchMove, {passive: false});
        }

        if (onTouchEnd) {
            ref.addEventListener('touchend', onTouchEnd);
        }
    }

    private removeTouchEventListeners(): void {
        const ref = this.getMonthsListRef();

        if (!ref) {
            return;
        }

        const {onTouchStart, onTouchEnd, onTouchMove} = this.props;

        if (onTouchStart) {
            ref.removeEventListener('touchstart', onTouchStart);
        }

        if (onTouchMove) {
            ref.removeEventListener('touchmove', onTouchMove);
        }

        if (onTouchEnd) {
            ref.removeEventListener('touchend', onTouchEnd);
        }
    }

    /* Refs manipulate */

    private setMonthsListRef = (monthListNode: HTMLDivElement): void => {
        this.monthsListNode = monthListNode;
    };

    private getMonthsListRef = (): HTMLDivElement | null => {
        return this.monthsListNode;
    };

    private setScrolledDayNodeRef = (dayNode: HTMLDivElement | null): void => {
        this.scrolledDayNode = dayNode;
    };

    private getScrolledDayNodeRef = (): HTMLDivElement | null => {
        return this.scrolledDayNode;
    };

    private setActiveDayRef = (node: HTMLDivElement | null): void => {
        this.activeDayRef = node;
    };

    private getActiveDayRef = (): HTMLDivElement | null => {
        return this.activeDayRef;
    };

    private checkScrollAnimateStatus(
        prevProps: ICalendarMonthsGridProps,
    ): void {
        const {scrollPartPosition, lastScrollSource} = this.props;

        if (scrollPartPosition !== prevProps.scrollPartPosition) {
            if (lastScrollSource === ECalendarScrollSource.MonthsGrid) {
                this.setState({withoutScrollAnimation: true});
            } else {
                this.setState({withoutScrollAnimation: false});
            }
        }
    }

    private calculateMonthsListMaxScrollTop(): void {
        const monthsListNode = this.getMonthsListRef();

        if (!monthsListNode) {
            return;
        }

        const {
            scrollHeight: monthsListScrollHeight,
            offsetHeight: monthsListOffsetHeight,
        } = monthsListNode;

        this.setState({
            maxScrollTop: monthsListScrollHeight - monthsListOffsetHeight,
        });
    }

    private getCalendarDayHeight(): ECalendarDayHeight {
        if (this.props.deviceType.isMobile) {
            return ECalendarDayHeight.MOBILE;
        }

        return ECalendarDayHeight.DESKTOP;
    }

    private getMonthHeight = (index: number): number => {
        const {months} = this.props;
        const month = months[index];
        const labelHeight = canMoveMonthLabel(month) ? 0 : MONTH_TITLE_HEIGHT;
        const rowsCount = month.monthDays.length / 7;

        return rowsCount * this.getCalendarDayHeight() + labelHeight;
    };

    // Не используем hoveredDate напрямую из props, чтобы не вызывать лишних
    // перерендеров у компонентов, на которые не влияет изменение hoveredDate
    private getHoveredDate(
        monthNum: ICalendarMonth['month'],
    ): undefined | Date {
        const {hoveredDate, startDate, endDate, calendarType} = this.props;

        if (!hoveredDate || hoveredDate.getMonth() === monthNum) {
            return hoveredDate;
        }

        // При выделении диапазона нам необходимо перерисовать все месяцы,
        // которые попадают в диапазон
        if (calendarType === ECalendarType.StartDate && !startDate && endDate) {
            return monthNum >= hoveredDate.getMonth() &&
                monthNum <= endDate.getMonth()
                ? hoveredDate
                : undefined;
        } else if (
            calendarType === ECalendarType.EndDate &&
            !endDate &&
            startDate
        ) {
            return monthNum <= hoveredDate.getMonth() &&
                monthNum >= startDate.getMonth()
                ? hoveredDate
                : undefined;
        }

        return undefined;
    }

    private scrollIntoScrolledDay(): void {
        const {canScrollIntoSelectedDate} = this.props;
        const scrolledDayNode = this.getScrolledDayNodeRef();

        if (scrolledDayNode && canScrollIntoSelectedDate) {
            this.scrollToNode(scrolledDayNode);
        }
    }

    private scrollToActiveDate = (): void => {
        const activeDayNode = this.getActiveDayRef();

        if (activeDayNode) {
            this.scrollToNode(activeDayNode);
        }
    };

    private scrollToNode(node: HTMLDivElement): void {
        /* При монтировании компонента у списка месяцев в первый момент
         * offsetHeight и scrollHeight равны, что приводит к maxScrollTop=Infinity
         * rAF для вывода из синхронного потока
         */
        requestAnimationFrame(() => {
            const {onChangeScrollPosition} = this.props;
            const {maxScrollTop} = this.state;
            const monthsNode = this.getMonthsListRef();

            if (node && monthsNode) {
                const nodePosition = node.getBoundingClientRect();
                const monthsPosition = monthsNode.getBoundingClientRect();
                const offsetTop =
                    nodePosition.top -
                        (monthsPosition.top - monthsNode.scrollTop) || 0;
                const scrollPartPosition = offsetTop / maxScrollTop;

                onChangeScrollPosition({
                    scrollPartPosition,
                    source: ECalendarScrollSource.MonthsGridMount,
                });
            }
        });
    }

    private changeScrolledDayNode = ({
        dayNode,
        type,
    }: {
        dayNode: HTMLDivElement;
        type: ECalendarType;
    }): void => {
        const scrolledDayNode = this.getScrolledDayNodeRef();

        if (type === ECalendarType.StartDate || !scrolledDayNode) {
            this.setScrolledDayNodeRef(dayNode);
        }
    };

    private flushScrollPosition = (scrollPartPosition: number): void => {
        const {onChangeScrollPosition} = this.props;

        onChangeScrollPosition({
            scrollPartPosition,
            source: ECalendarScrollSource.MonthsGrid,
        });
    };

    private handleScrollMonths: UIEventHandler<HTMLDivElement> = e => {
        const {maxScrollTop, withoutScrollAnimation} = this.state;

        if (withoutScrollAnimation) {
            const {scrollTop} = e.currentTarget;
            const scrollPartPosition = maxScrollTop && scrollTop / maxScrollTop;

            this.flushScrollPosition(scrollPartPosition);
        }
    };

    private renderWeekdays(): React.ReactNode {
        const {weekdays, weekdaysClassName} = this.props;

        return (
            <CalendarWeekdays
                className={cx('weekdays', weekdaysClassName)}
                weekdays={weekdays}
            />
        );
    }

    private renderMonthOrPlaceholder = (
        month: ICalendarMonth,
        index: number,
    ): React.ReactNode => {
        const {startDate, endDate, months, scrollPartPosition, nowDate} =
            this.props;
        const scrollToMonth = Math.ceil(months.length * scrollPartPosition) + 1;
        const monthsShouldBeRenderUpTo = Math.max(
            ...[
                startDate
                    ? (startDate.getFullYear() - nowDate.getFullYear()) * 12 +
                      startDate.getMonth() +
                      1
                    : 0,
                endDate
                    ? (endDate.getFullYear() - nowDate.getFullYear()) * 12 +
                      endDate.getMonth() +
                      1
                    : 0,
                MIN_MONTH_TO_RENDER,
                scrollToMonth,
            ],
        );

        return (
            <InView key={index} triggerOnce>
                {({ref, inView}): React.ReactNode => (
                    <div ref={ref}>
                        {inView || index < monthsShouldBeRenderUpTo
                            ? this.renderMonth(index)
                            : this.renderPlaceholder(index)}
                    </div>
                )}
            </InView>
        );
    };

    private renderPlaceholder = (index: number): React.ReactNode => (
        <div
            className={cx('monthPlaceholder')}
            style={{height: this.getMonthHeight(index)}}
        />
    );

    private renderMonth = (index: number): React.ReactNode => {
        const {
            deviceType,
            months,
            startDate,
            endDate,
            nowDate,
            minDate,
            maxDate,
            isRangeSelected,
            calendarType,
            prices,
            dayClassName,
            onDayClick,
            onDayMouseLeave,
            onDayMouseEnter,
        } = this.props;

        return (
            <CalendarMonth
                className={cx('month')}
                deviceType={deviceType}
                month={months[index]}
                startDate={startDate}
                endDate={endDate}
                nowDate={nowDate}
                minDate={minDate}
                maxDate={maxDate}
                dayClassName={dayClassName}
                hoveredDate={this.getHoveredDate(months[index].month)}
                isRangeSelected={isRangeSelected}
                calendarType={calendarType}
                onDayClick={onDayClick}
                onDayMouseLeave={onDayMouseLeave}
                onDayMouseEnter={onDayMouseEnter}
                setActiveDayRef={this.setActiveDayRef}
                setScrolledDayNode={this.changeScrolledDayNode}
                prices={prices}
                {...prepareQaAttributes(this.props)}
            />
        );
    };

    private renderMonths(): React.ReactNode {
        const {months} = this.props;

        return (
            <div
                className={cx('months')}
                onScroll={this.handleScrollMonths}
                ref={this.setMonthsListRef}
            >
                {months.map(this.renderMonthOrPlaceholder)}
            </div>
        );
    }

    private onSpringRest = (): void => {
        this.setState({withoutScrollAnimation: true});
    };

    private renderCalendarScrollController(): React.ReactNode {
        const {scrollPartPosition} = this.props;
        const {maxScrollTop, withoutScrollAnimation} = this.state;
        const monthsListNode = this.getMonthsListRef();
        const scrollTop = scrollPartPosition * maxScrollTop;

        return (
            <CalendarScrollController
                scrollTop={scrollTop}
                scrollNode={monthsListNode}
                onSpringRest={this.onSpringRest}
                withoutAnimation={withoutScrollAnimation}
            />
        );
    }

    render(): React.ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'root',
                    deviceModDesktop('root', deviceType),
                    className,
                )}
            >
                {this.renderWeekdays()}
                {this.renderMonths()}
                {this.renderCalendarScrollController()}
            </div>
        );
    }
}

export default CalendarMonthsGrid;
