import React, {PureComponent} from 'react';
import _noop from 'lodash/noop';
import {subscribe, unsubscribe} from 'pubsub-js';
import {InView} from 'react-intersection-observer';

import {CALENDAR_EVENTS} from 'components/Calendar/constants/constants';

import {IWithClassName} from 'types/withClassName';
import {ICalendarMonth, TWeekDayObject} from 'components/Calendar/types';
import {TCalendarPrices} from 'types/common/calendarPrice/ICalendarPrice';
import {EDatePickerSelectionMode} from 'components/DatePicker/types';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModDesktop} from 'utilities/stylesUtils';
import sortDates from 'utilities/dateUtils/sortDates';
import getCalendarDayHeight from 'components/Calendar/utilities/getCalendarDayHeight';
import getMonthWeeks from 'components/Calendar/utilities/getMonthWeeks';
import findMonthDay, {
    TDayInfo,
} from 'components/Calendar/utilities/findMonthDay';

import CalendarMonth from 'components/Calendar/components/CalendarMonth/CalendarMonthNew';
import CalendarWeekdays from 'components/Calendar/components/CalendarWeekdays/CalendarWeekdays';
import CalendarScrollController from 'components/Calendar/components/CalendarScrollController/CalendarScrollController';

import cx from './CalendarMonthsGridNew.scss';

const MIN_MONTH_TO_RENDER = 3;
const MONTH_TITLE_HEIGHT = 32;
const AVERAGE_MONTH_ELEMENT_HEIGHT = 318;

export type TVisibleMonth = {
    index: number;
    month: ICalendarMonth;
};

interface ICalendarMonthsGridProps extends IWithClassName, IWithQaAttributes {
    deviceType: IDevice;
    dayClassName: string;
    weekdaysClassName: string;
    months: ICalendarMonth[];
    weekdays: TWeekDayObject[];
    nowDate: Date;
    minDate: Date;
    maxDate: Date;
    startDate: Date;
    endDate: Date;
    hoveredDate?: Date;
    isRangeSelected: boolean;
    canScrollIntoSelectedDate: boolean;
    selectionMode: EDatePickerSelectionMode;
    canSelectRange?: boolean;
    prices?: TCalendarPrices;
    onDayClick: (selectedDate: Date, dayNode: HTMLDivElement) => void;
    setHoveredDate: (hoveredDate: Date | null) => void;

    onTouchStart?: (event: TouchEvent) => void;
    onTouchMove?: (event: TouchEvent) => void;
    onTouchEnd?: (event: TouchEvent) => void;
    onVisibleMonthChange?: (visibleMonth: TVisibleMonth) => void;
}

interface ICalendarMonthsGridState {
    animationInProgress: boolean;
    scrollPosition: number;
    lastScrollToMonth: number;
}

class CalendarMonthsGrid extends PureComponent<
    ICalendarMonthsGridProps,
    ICalendarMonthsGridState
> {
    private monthsListNode: HTMLDivElement | null = null;
    private rootNode: HTMLDivElement | null = null;
    /**
     * Во время анимации подскролла не отправляем изменение о наведенной дате, а сохраняем локально, на конец анимации отправляем наверх
     * undefined - не менялась наведенная дата, null - убрали наведение на дату, Date - навели на другую дату
     */
    private lastHoveredDate: Date | null | undefined;

    static defaultProps = {
        className: '',
        dayClassName: '',
        weekdaysClassName: '',
        months: [],
        weekdays: [],
        onChangeScrollPosition: _noop,
        startDate: null,
        endDate: null,
        headerNode: null,
        canScrollIntoSelectedDate: true,
        onDayClick: _noop,
        onDayMouseLeave: _noop,
        onDayMouseEnter: _noop,
    };

    constructor(props: ICalendarMonthsGridProps) {
        super(props);

        this.state = {
            animationInProgress: false,
            scrollPosition: 0,
            lastScrollToMonth: 0,
        };
    }

    componentDidMount(): void {
        this.subscribe();
        this.scrollIntoScrolledDay();

        this.addTouchEventListeners();
    }

    componentWillUnmount(): void {
        this.unsubscribe();
        this.removeTouchEventListeners();
    }

    subscribe(): void {
        subscribe(
            CALENDAR_EVENTS.SCROLL_TO_ACTIVE_DATE,
            this.scrollToActiveDate,
        );
    }

    unsubscribe(): void {
        unsubscribe(CALENDAR_EVENTS.SCROLL_TO_ACTIVE_DATE);
    }

    addTouchEventListeners(): void {
        const ref = this.getMonthsListRef();

        if (!ref) {
            return;
        }

        const {onTouchStart, onTouchEnd, onTouchMove} = this.props;

        // Добавляем через eventListeners, вместо системы событий реакта,
        // Так как нет возможности указать {passive: false}
        if (onTouchStart) {
            ref.addEventListener('touchstart', onTouchStart);
        }

        if (onTouchMove) {
            ref.addEventListener('touchmove', onTouchMove, {passive: false});
        }

        if (onTouchEnd) {
            ref.addEventListener('touchend', onTouchEnd);
        }
    }

    removeTouchEventListeners(): void {
        const ref = this.getMonthsListRef();

        if (!ref) {
            return;
        }

        const {onTouchStart, onTouchEnd, onTouchMove} = this.props;

        if (onTouchStart) {
            ref.removeEventListener('touchstart', onTouchStart);
        }

        if (onTouchMove) {
            ref.removeEventListener('touchmove', onTouchMove);
        }

        if (onTouchEnd) {
            ref.removeEventListener('touchend', onTouchEnd);
        }
    }

    /* Refs manipulate */

    setRootRef = (rootNode: HTMLDivElement): void => {
        this.rootNode = rootNode;
    };

    setMonthsListRef = (monthListNode: HTMLDivElement): void => {
        this.monthsListNode = monthListNode;
    };

    getMonthsListRef = (): HTMLDivElement | null => {
        return this.monthsListNode;
    };

    getMonthHeight = (index: number): number => {
        const {months, deviceType} = this.props;
        const dayHeight = getCalendarDayHeight(deviceType.isMobile);
        const month = months[index];
        const labelHeight = MONTH_TITLE_HEIGHT;
        const rowsCount = month.monthDays.length / 7;

        return rowsCount * dayHeight + labelHeight;
    };

    // Не используем hoveredDate напрямую из props, чтобы не вызывать лишних
    // перерендеров у компонентов, на которые не влияет изменение hoveredDate
    getHoveredDate(monthNum: ICalendarMonth['month']): undefined | Date {
        const {hoveredDate, startDate, endDate} = this.props;

        if (!hoveredDate || hoveredDate.getMonth() === monthNum) {
            return hoveredDate;
        }

        // При выделении диапазона нам необходимо перерисовать все месяцы,
        // которые попадают в диапазон
        if (startDate && !endDate) {
            const [left, right] = sortDates(startDate, hoveredDate);

            return monthNum <= right.getMonth() && monthNum >= left.getMonth()
                ? hoveredDate
                : undefined;
        }

        return undefined;
    }

    private scrollIntoScrolledDay(): void {
        const {canScrollIntoSelectedDate} = this.props;
        const dayToScroll = findMonthDay(d => d.isScrolledDate, this.props);

        if (dayToScroll && canScrollIntoSelectedDate) {
            this.scrollToDate(dayToScroll);
        }
    }

    private scrollToActiveDate = (): void => {
        const dayToScroll = findMonthDay(
            d => d.isStartDate || d.isEndDate,
            this.props,
        );

        if (dayToScroll) {
            this.scrollToDate(dayToScroll);
        }
    };

    private getNodeHeightInViewportByIndex = (
        index: number,
    ): number | undefined => {
        if (!this.rootNode) {
            return undefined;
        }

        if (!this.monthsListNode) {
            return undefined;
        }

        const rootHeight = this.rootNode.getBoundingClientRect().height;
        const node = this.monthsListNode.children[index] as HTMLDivElement;

        if (node === undefined) {
            return undefined;
        }

        const top = this.getNodeTopByIndex(index);
        const nodeHeight = node.getBoundingClientRect().height;

        if (top === undefined) {
            return undefined;
        }

        const {startDate, endDate, nowDate, months} = this.props;
        const month = months[index];
        // текущему месяцу даем больший вес,
        // чтобы он считался находямимся вовьюпорте,
        // в том случае если текущая дата выпадает на конец месяца
        // и следующий месяц занимает чуть больше места
        const add =
            (startDate || endDate || nowDate).getMonth() === month.month
                ? 150
                : 0;

        return this.intersection(0, rootHeight, top, top + nodeHeight) + add;
    };

    private intersection(
        x1: number,
        x2: number,
        x3: number,
        x4: number,
    ): number {
        if (x2 < x3 || x4 < x1) {
            return 0;
        }

        if (x1 < x3) {
            return x2 - x3;
        }

        return x4 - x1;
    }

    private getNodeTopByIndex = (index: number): number | undefined => {
        if (!this.monthsListNode) {
            return undefined;
        }

        const offset = this.getNodeOffsetByIndex(index);

        if (offset === undefined) {
            return undefined;
        }

        return offset - this.monthsListNode.scrollTop;
    };

    private getNodeOffsetByIndex = (index: number): number | undefined => {
        if (!this.monthsListNode) {
            return undefined;
        }

        const node = this.monthsListNode.children[index] as HTMLDivElement;

        if (!node) {
            return undefined;
        }

        return this.getNodeOffset(node);
    };

    private getNodeOffset = (node: HTMLDivElement): number | undefined => {
        const monthsNode = this.getMonthsListRef();

        if (!node || !monthsNode) {
            return undefined;
        }

        const nodePosition = node.getBoundingClientRect();
        const monthsPosition = monthsNode.getBoundingClientRect();
        const offsetTop =
            nodePosition.top - (monthsPosition.top - monthsNode.scrollTop) || 0;

        return offsetTop;
    };

    scrollToDate = ({day, monthIndex}: TDayInfo): void => {
        const monthsNode = this.getMonthsListRef();

        if (!monthsNode) {
            return;
        }

        const {deviceType, onVisibleMonthChange, months} = this.props;
        const monthWeeks = getMonthWeeks(months[monthIndex].monthDays);
        const weekIndex = day
            ? monthWeeks.findIndex(w => w.some(d => d && d.day === day.day))
            : 0;
        const node = monthsNode.children[monthIndex] as HTMLDivElement;
        const dayHeight = getCalendarDayHeight(deviceType.isMobile);
        const offset = this.getNodeOffset(node);

        if (offset === undefined) {
            return;
        }

        this.setState({
            animationInProgress: true,
            scrollPosition: offset + weekIndex * dayHeight - 2,
            lastScrollToMonth: monthIndex,
        });

        onVisibleMonthChange?.({
            index: monthIndex,
            month: months[monthIndex],
        });
    };

    private handleDayClick = (
        selectedDate: Date,
        dayNode: HTMLDivElement,
    ): void => {
        const {selectionMode, onDayClick} = this.props;

        onDayClick(selectedDate, dayNode);

        if (selectionMode === EDatePickerSelectionMode.START_DATE) {
            const offset = this.getNodeOffset(dayNode);

            if (offset !== undefined) {
                this.setState({
                    animationInProgress: true,
                    scrollPosition: offset - MONTH_TITLE_HEIGHT,
                });
            }
        }
    };

    private handleDayMouseLeave = (): void => {
        const {setHoveredDate} = this.props;

        if (this.state.animationInProgress) {
            this.lastHoveredDate = null;
        } else {
            setHoveredDate(null);
        }
    };

    private handleDayMouseEnter = (hoveredDate: Date): void => {
        const {setHoveredDate} = this.props;

        if (this.state.animationInProgress) {
            this.lastHoveredDate = hoveredDate;
        } else {
            setHoveredDate(hoveredDate);
        }
    };

    private handleScroll = (): void => {
        if (!this.monthsListNode) {
            return;
        }

        const candidateIndex = Math.ceil(
            this.monthsListNode.scrollTop / AVERAGE_MONTH_ELEMENT_HEIGHT,
        );

        this.tryCallVisibleMonthChange(candidateIndex);
    };

    private handleAnimationEnd = (): void => {
        const {setHoveredDate} = this.props;
        const {lastScrollToMonth} = this.state;

        this.setState({animationInProgress: false}, () =>
            this.tryCallVisibleMonthChange(lastScrollToMonth),
        );

        if (this.lastHoveredDate !== undefined) {
            setHoveredDate(this.lastHoveredDate);
        }

        this.lastHoveredDate = undefined;
    };

    private tryCallVisibleMonthChange(candidateIndex: number): void {
        if (!this.monthsListNode) {
            return;
        }

        const {animationInProgress} = this.state;

        if (animationInProgress) {
            return;
        }

        const {months, onVisibleMonthChange} = this.props;

        const prev = this.findNodeInViewportIndex(candidateIndex, -1);
        const next = this.findNodeInViewportIndex(candidateIndex, 1);

        if (prev !== undefined && next !== undefined) {
            if (prev.size > next.size) {
                onVisibleMonthChange?.({
                    index: prev.index,
                    month: months[prev.index],
                });
            } else {
                onVisibleMonthChange?.({
                    index: next.index,
                    month: months[next.index],
                });
            }
        } else if (prev !== undefined) {
            onVisibleMonthChange?.({
                index: prev.index,
                month: months[prev.index],
            });
        } else if (next !== undefined) {
            onVisibleMonthChange?.({
                index: next.index,
                month: months[next.index],
            });
        }
    }

    private findNodeInViewportIndex(
        index: number,
        direction: number,
    ): {index: number; size: number} | undefined {
        if (!this.monthsListNode) {
            return undefined;
        }

        let resultIndex = index;
        let resultViewport = this.getNodeHeightInViewportByIndex(resultIndex);

        if (resultViewport === undefined) {
            return undefined;
        }

        for (
            let candidateIndex = index;
            candidateIndex >= 0 &&
            candidateIndex < this.monthsListNode.children.length;
            candidateIndex += direction
        ) {
            const candidateViewport =
                this.getNodeHeightInViewportByIndex(candidateIndex);

            if (
                candidateViewport !== undefined &&
                resultViewport < candidateViewport
            ) {
                resultIndex = candidateIndex;
                resultViewport = candidateViewport;
            }
        }

        if (resultViewport > 0) {
            return {index: resultIndex, size: resultViewport};
        }

        return undefined;
    }

    private renderWeekdays(): React.ReactNode {
        const {weekdays, weekdaysClassName} = this.props;

        return (
            <CalendarWeekdays
                className={cx('weekdays', weekdaysClassName)}
                weekdays={weekdays}
            />
        );
    }

    private renderMonthOrPlaceholder = (
        month: ICalendarMonth,
        index: number,
    ): React.ReactNode => {
        const {lastScrollToMonth} = this.state;
        const {startDate, endDate, nowDate} = this.props;
        const monthsShouldBeRenderUpTo = Math.max(
            ...[
                startDate
                    ? (startDate.getFullYear() - nowDate.getFullYear()) * 12 +
                      startDate.getMonth() +
                      1
                    : 0,
                endDate
                    ? (endDate.getFullYear() - nowDate.getFullYear()) * 12 +
                      endDate.getMonth() +
                      1
                    : 0,
                MIN_MONTH_TO_RENDER,
                lastScrollToMonth,
            ],
        );

        return (
            <InView key={index} triggerOnce>
                {({ref, inView}): React.ReactNode => (
                    <div ref={ref}>
                        {inView || index < monthsShouldBeRenderUpTo
                            ? this.renderMonth(index)
                            : this.renderPlaceholder(index)}
                    </div>
                )}
            </InView>
        );
    };

    private renderPlaceholder = (index: number): React.ReactNode => (
        <div
            className={cx('monthPlaceholder')}
            style={{height: this.getMonthHeight(index)}}
        />
    );

    private renderMonth = (index: number): React.ReactNode => {
        const {
            deviceType,
            months,
            startDate,
            endDate,
            nowDate,
            minDate,
            maxDate,
            isRangeSelected,
            prices,
            dayClassName,
            canSelectRange,
        } = this.props;

        return (
            <CalendarMonth
                className={cx('month')}
                deviceType={deviceType}
                month={months[index]}
                startDate={startDate}
                endDate={endDate}
                nowDate={nowDate}
                minDate={minDate}
                maxDate={maxDate}
                dayClassName={dayClassName}
                hoveredDate={this.getHoveredDate(months[index].month)}
                isRangeSelected={isRangeSelected}
                onDayClick={this.handleDayClick}
                onDayMouseLeave={this.handleDayMouseLeave}
                onDayMouseEnter={this.handleDayMouseEnter}
                canSelectRange={canSelectRange}
                prices={prices}
                {...prepareQaAttributes(this.props)}
            />
        );
    };

    private renderMonths(): React.ReactNode {
        const {months} = this.props;

        return (
            <div
                className={cx('months')}
                ref={this.setMonthsListRef}
                onScroll={this.handleScroll}
            >
                {months.map(this.renderMonthOrPlaceholder)}
            </div>
        );
    }

    private renderCalendarScrollController(): React.ReactNode {
        const {scrollPosition, animationInProgress} = this.state;
        const monthsListNode = this.getMonthsListRef();

        return (
            <CalendarScrollController
                scrollTop={scrollPosition}
                scrollNode={monthsListNode}
                onSpringRest={this.handleAnimationEnd}
                withoutAnimation={!animationInProgress}
            />
        );
    }

    render(): React.ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(
                    'root',
                    deviceModDesktop('root', deviceType),
                    className,
                )}
                ref={this.setRootRef}
            >
                {this.renderWeekdays()}
                {this.renderMonths()}
                {this.renderCalendarScrollController()}
            </div>
        );
    }
}

export default CalendarMonthsGrid;
