import React, {createRef, PureComponent} from 'react';
import {Spring, animated} from 'react-spring';

import {IWithClassName} from 'types/withClassName';
import {ECalendarType, ICalendarMonth} from 'components/Calendar/types';

import findMonthDay from 'components/Calendar/utilities/findMonthDay';

import CalendarScrollController from '../CalendarScrollController/CalendarScrollController';

import cx from './CalendarMonthsListNew.scss';

const ACTIVE_MONTHS_SCROLL_HEIGHT = 32;
const ACTIVE_MONTHS_SCROLL_OFFSET = 8;

interface ICalendarMonthsListProps extends IWithClassName {
    months: ICalendarMonth[];
    highlightMonth: number;
    scrollToMonth: (index: number) => void;

    nowDate: Date;
    minDate: Date;
    maxDate: Date;
    endDate: Date;
    startDate: Date;
    hoveredDate?: Date;
    calendarType?: ECalendarType;
    newExperimentalCalendar?: boolean;
    canSelectRange?: boolean;
}

interface ICalendarMonthsListState {
    monthsListHeight: number;
    scrollTo?: number;
    scrollToMothsIndex?: number;
}

type TMonthsListNode = HTMLDivElement | null;

class CalendarMonthsListNew extends PureComponent<
    ICalendarMonthsListProps,
    ICalendarMonthsListState
> {
    static defaultProps = {
        className: '',
        months: [],
        highlightMonth: 0,
        endDate: null,
        startDate: null,
    };
    monthsListNode: TMonthsListNode = null;
    rootRef = createRef<HTMLDivElement>();

    constructor(props: ICalendarMonthsListProps) {
        super(props);

        this.state = {
            monthsListHeight: 0,
        };
    }

    componentDidMount(): void {
        this.calculateAndSetMonthsListHeight();
    }

    componentDidUpdate(prevProps: ICalendarMonthsListProps): void {
        const {highlightMonth} = this.props;

        if (prevProps.highlightMonth !== highlightMonth) {
            this.scrollToHighlightMonth();
        }
    }

    private scrollToScrolledMonth = (): void => {
        const dayToScroll = findMonthDay(d => d.isScrolledDate, this.props);

        if (dayToScroll) {
            const firstMonthOfTheYear = this.props.months.findIndex(
                m => m.year === dayToScroll.day?.year,
            );

            this.scrollToMonth(firstMonthOfTheYear);
        }
    };

    private scrollToHighlightMonth = (): void => {
        if (!this.rootRef.current) {
            return;
        }

        if (this.isAnimationInProgress()) {
            return;
        }

        const {highlightMonth} = this.props;
        const {scrollTop} = this.rootRef.current;
        const {height} = this.rootRef.current.getBoundingClientRect();

        const activeMonthTop = this.getMonthScrollPosition(highlightMonth);
        const activeMonthShift = height - ACTIVE_MONTHS_SCROLL_HEIGHT;

        if (scrollTop > activeMonthTop) {
            this.setState({
                scrollTo: activeMonthTop,
                scrollToMothsIndex: highlightMonth,
            });
        } else if (scrollTop + activeMonthShift < activeMonthTop) {
            this.setState({
                scrollTo:
                    activeMonthTop -
                    activeMonthShift +
                    ACTIVE_MONTHS_SCROLL_OFFSET,
                scrollToMothsIndex: highlightMonth,
            });
        }
    };

    scrollToMonth = (monthIndex: number): void => {
        if (!this.rootRef.current) {
            return;
        }

        if (this.isAnimationInProgress()) {
            return;
        }

        const activeMonthTop = this.getMonthScrollPosition(monthIndex);

        this.setState({
            scrollTo: activeMonthTop + ACTIVE_MONTHS_SCROLL_OFFSET,
            scrollToMothsIndex: monthIndex,
        });
    };

    private isAnimationInProgress(): boolean {
        return this.state.scrollTo !== undefined;
    }

    calculateAndSetMonthsListHeight(): void {
        const monthsListNode = this.getMonthsListRef();

        if (!monthsListNode) {
            return;
        }

        const {height: monthsListHeight} =
            monthsListNode.getBoundingClientRect();

        this.setState({monthsListHeight}, this.scrollToScrolledMonth);
    }

    getMonthScrollPosition(monthIndex: number): number {
        const {monthsListHeight} = this.state;
        const {months} = this.props;

        const step = monthsListHeight / months.length;

        return step * monthIndex;
    }

    setMonthsListRef = (monthsListNode: TMonthsListNode): void => {
        this.monthsListNode = monthsListNode;
    };

    getMonthsListRef = (): TMonthsListNode => {
        return this.monthsListNode;
    };

    handleAnimationEnd = (): void => {
        this.setState({scrollTo: undefined}, () => {
            if (this.props.highlightMonth === this.state.scrollToMothsIndex) {
                return;
            }

            this.scrollToHighlightMonth();
        });
    };

    renderActiveMonthsScrollPosition(): React.ReactNode {
        const {highlightMonth} = this.props;
        const scrollTopPosition = this.getMonthScrollPosition(highlightMonth);

        return (
            <Spring to={{transform: `translateY(${scrollTopPosition}px)`}}>
                {({transform}): React.ReactElement => (
                    <animated.div
                        className={cx('activeMonthsScroll')}
                        style={{
                            top: ACTIVE_MONTHS_SCROLL_OFFSET,
                            height: ACTIVE_MONTHS_SCROLL_HEIGHT,
                            transform,
                        }}
                    />
                )}
            </Spring>
        );
    }

    renderMonth = (
        {monthLabel, month, year}: ICalendarMonth,
        monthIndex: number,
    ): React.ReactNode => {
        const isFirstMonth = month === 0;
        const {scrollToMonth} = this.props;

        return (
            <div
                key={monthIndex}
                className={cx('month')}
                onClick={(): void => scrollToMonth(monthIndex)}
            >
                {monthLabel}
                {isFirstMonth && ` ${year}`}
            </div>
        );
    };

    renderMonthsList(): React.ReactNode {
        const {months} = this.props;

        return (
            <div className={cx('monthsList')} ref={this.setMonthsListRef}>
                {months.map(this.renderMonth)}
            </div>
        );
    }

    renderCalendarScrollController(): React.ReactNode {
        const {scrollTo} = this.state;

        return (
            <CalendarScrollController
                scrollTop={scrollTo || 0}
                scrollNode={this.rootRef.current}
                onSpringRest={this.handleAnimationEnd}
                withoutAnimation={!this.isAnimationInProgress()}
            />
        );
    }

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div
                className={cx('calendarMonthsList', className)}
                ref={this.rootRef}
            >
                {this.renderActiveMonthsScrollPosition()}
                {this.renderMonthsList()}
                {this.renderCalendarScrollController()}
            </div>
        );
    }
}

export default CalendarMonthsListNew;
