import {
    Calendar as CalendarBase,
    TMonthNumber,
    TCalendarDateObject,
} from 'calendar-base';
import _times from 'lodash/times';

import {
    MONTHS_IN_YEAR,
    DAYS_IN_WEEK,
} from 'components/Calendar/constants/constants';

import {ICalendarMonth} from 'components/Calendar/types';

import {getNow} from 'utilities/dateUtils';

import differenceInCalendarMonths from './differenceInCalendarMonths';
import getMonthLabelByIndex from './getMonthLabelByIndex';

type TOptions = {
    minDate: Date;
    maxDate: Date;
    weekStart?: number;
    siblingMonths?: boolean;
    cutPastWeeks?: boolean;
    cutDate?: Date;
};

export default (options: TOptions): ICalendarMonth[] => {
    const {
        minDate,
        maxDate,
        weekStart = 1,
        siblingMonths = false,
        cutPastWeeks = false,
        cutDate = new Date(getNow()),
    } = options;

    const calendarBaseInstance = new CalendarBase({weekStart, siblingMonths});

    if (minDate && maxDate) {
        const minDateYear = minDate.getFullYear();
        const minDateMonths = minDate.getMonth();

        const monthsTotal = differenceInCalendarMonths(maxDate, minDate) + 1;

        const calendarMonths = _times(monthsTotal, monthsIndex => {
            const month = minDateMonths + monthsIndex;
            const fullYears = Math.floor(month / MONTHS_IN_YEAR);
            const currentYear = minDateYear + fullYears;
            const currentMonth = (month -
                fullYears * MONTHS_IN_YEAR) as TMonthNumber;
            const monthLabel = getMonthLabelByIndex({monthIndex: currentMonth});

            let monthDays = calendarBaseInstance.getCalendar(
                currentYear,
                currentMonth,
            );

            if (cutPastWeeks && monthsIndex === 0) {
                const cutWeekDay = cutDate.getUTCDay();
                const dayOfWeekTransformed = cutWeekDay === 0 ? 7 : cutWeekDay;
                const checkingDay = cutDate.getUTCDate() - dayOfWeekTransformed;

                monthDays = monthDays.reduce<TCalendarDateObject[]>(
                    (prev, monthDay, index) => {
                        if (
                            // cut weeks with no enabled days
                            (monthDay &&
                                cutDate.getUTCFullYear() === monthDay.year &&
                                cutDate.getUTCMonth() === monthDay.month &&
                                monthDay.day <= checkingDay) ||
                            // cut 'false' days if cut week isn't first in cut month
                            (!monthDay &&
                                cutDate.getUTCDate() > dayOfWeekTransformed &&
                                index < DAYS_IN_WEEK)
                        ) {
                            return prev;
                        }

                        prev.push(monthDay);

                        return prev;
                    },
                    [],
                );
            }

            return {
                monthDays,
                monthLabel,
                year: currentYear,
                month: currentMonth,
            };
        });

        return calendarMonths;
    }

    return [];
};
