import React, {FunctionComponent, ReactNode} from 'react';
import classnames from 'classnames';

import {ECardWithDeviceLayoutVariation} from 'components/CardWithDeviceLayout/types/ECardWithDeviceLayoutVariation';
import {ICardWithDeviceLayoutOptionalPaddings} from 'components/CardWithDeviceLayout/types/ICardWithDeviceLayoutOptionalPaddings';

import {usePadding} from 'utilities/hooks/usePadding/usePadding';
import {IWithQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import getPaddingSizes from 'components/CardWithDeviceLayout/utilities/getPaddingSizes';

import Card, {ICardProps} from 'components/Card/Card';
import Box from 'components/Box/Box';

export interface ICardWithDeviceLayoutProps
    extends Pick<
            ICardProps,
            | 'between'
            | 'above'
            | 'below'
            | 'className'
            | 'withHover'
            | 'shadow'
            | 'radius'
            | 'ref'
            | 'background'
            | 'onClick'
        >,
        ICardWithDeviceLayoutOptionalPaddings,
        IWithQaAttributes {
    variation?: ECardWithDeviceLayoutVariation;
    sectioned?: boolean;
    children: ReactNode;
}

const Section: FunctionComponent<ICardWithDeviceLayoutProps> = ({
    className,
    children,
    variation = ECardWithDeviceLayoutVariation.MAIN,
    paddingTop,
    paddingBottom,
    ...rest
}) => {
    const sizes = getPaddingSizes(variation, {paddingTop, paddingBottom});
    const paddingClassNames = usePadding(sizes);

    return (
        <Box className={classnames(paddingClassNames, className)} {...rest}>
            {children}
        </Box>
    );
};

// @ts-ignore
const CardWithDeviceLayout: FunctionComponent<ICardWithDeviceLayoutProps> & {
    Section: typeof Section;
} = React.forwardRef<HTMLElement, ICardWithDeviceLayoutProps>(
    (
        {
            className,
            children,
            variation = ECardWithDeviceLayoutVariation.MAIN,
            sectioned = false,
            below,
            paddingTop,
            paddingBottom,
            shadow,
            ...rest
        },
        ref,
    ) => {
        const sizes = getPaddingSizes(variation, {paddingTop, paddingBottom});
        const paddingClassNames = usePadding(sizes);

        return (
            <Card
                className={classnames(
                    sectioned ? '' : paddingClassNames,
                    className,
                )}
                shadow={shadow ?? 'default'}
                below={below}
                ref={ref}
                {...rest}
            >
                {children}
            </Card>
        );
    },
);

CardWithDeviceLayout.displayName = 'CardWithDeviceLayout';
CardWithDeviceLayout.Section = Section;

export default CardWithDeviceLayout;
