import React, {Children, createRef, ReactNode, Component} from 'react';
import classNames from 'classnames';
import _once from 'lodash/once';
import _throttle from 'lodash/throttle';

import {BASE_SPACE} from 'constants/sizes';

import {IWithClassName} from 'src/types/withClassName';
import {ISize} from 'types/common/TImage';

import {
    getNextOffsetByRefs,
    getInvisibleRightIndex,
    getPrevOffsetByRefs,
    getInvisibleLeftIndex,
    getInvisibleRightIndexBySizes,
    getInitialOffsetByIndex,
    getInitialHasPrevByOffset,
    getInitialHasNextByOffset,
} from './utilities';

import Box, {TBoxSizes} from 'components/Box/Box';
import CarouselButton, {
    TArrowsType,
} from './components/CarouselButton/CarouselButton';

import cx from './Carousel.scss';

/* Constants */
const DEFAULT_SCROLL_CENTER_OPTIONS = {
    behavior: 'smooth' as 'smooth',
    inline: 'center' as 'center',
};

const SCROLL_THROTTLE_TIMEOUT = 100;

/* Component Types */
interface ICarouselState {
    scrollOffset: number;
    hasNext: boolean;
    hasPrev: boolean;
    containerHeight: number;
    /** Количество элементов в карусели */
    lazyLoadLimit: number;
}

type TCarouselType = 'full' | 'mini';

export interface ICarouselProps extends IWithClassName {
    children: ReactNode;

    // Styling
    height?: number;
    type?: TCarouselType;
    arrowsType?: TArrowsType;
    arrowsPosition?: 'center' | 'bottom';
    listClassName?: string;
    itemClassName?: string;
    buttonClassName?: string;
    gradient?: boolean;

    // Spacing
    spaceBetween?: TBoxSizes;
    /** На сколько пикселей будет виден предыдущий элемент при прокрутке влево */
    leftOffset?: number;
    /** На сколько пикселей будет виден следующий элемент при прокрутке вправо */
    rightOffset?: number;
    /** Отступ в начале карусели перед первым элементом */
    spaceBefore?: TBoxSizes;
    /** Отступ в конце карусели, после последнего элемента */
    spaceAfter?: TBoxSizes;

    // Controls
    scroll?: 'arrow' | 'native';
    hideArrows?: boolean;
    hideDisabledArrows?: boolean;

    // LazyLoading
    lazyLoad?: boolean;
    lazyLoadSizes?: (ISize | undefined)[];
    lazyPlaceholder?: ReactNode;

    onLastItemVisible?: (index: number) => void;
    getItemWidth?: (index: number) => number | string | undefined;

    // Initial visible element for fixed width Carousels
    initialVisibleIndex?: number;

    /* Handlers */
    onPrevButtonClick?: () => void;
    onNextButtonClick?: () => void;
    onScrollContent?: (index?: number) => void;
}

/**
 * Компонент карусели любых элементов.
 * Отображает скрытый (частично/полностью) элемент, со сдвигом
 */
class Carousel extends Component<ICarouselProps, ICarouselState> {
    private readonly containerRef = createRef<HTMLDivElement>();
    private readonly listRef = createRef<HTMLDivElement>();
    private itemRefs: (HTMLDivElement | null)[] = [];
    private handleLastItemVisible =
        this.props.onLastItemVisible && _once(this.props.onLastItemVisible);

    private handleScroll = _throttle(
        () => {
            const {lazyLoad, children, onScrollContent} = this.props;
            const {lazyLoadLimit} = this.state;

            const childrenLength = Children.count(children);
            const index = this.getInvisibleRightIndex() || childrenLength - 1;

            if (lazyLoad && index + 1 >= lazyLoadLimit) {
                this.setState({
                    lazyLoadLimit: index + 1,
                });
            }

            if (
                this.handleLastItemVisible &&
                (index === 0 || index + 1 === childrenLength)
            ) {
                this.handleLastItemVisible(index);
            }

            if (onScrollContent) {
                onScrollContent();
            }
        },
        SCROLL_THROTTLE_TIMEOUT,
        {trailing: true},
    );

    static readonly defaultProps: Partial<ICarouselProps> = {
        type: 'full',

        spaceBetween: 0,
        leftOffset: 0,
        rightOffset: 0,

        scroll: 'arrow',
    };

    readonly state: ICarouselState = {
        scrollOffset: 0,
        containerHeight: 1,
        hasNext: false,
        hasPrev: false,
        lazyLoadLimit: 0,
    };

    componentDidMount(): void {
        const scrollOffset = getInitialOffsetByIndex(
            this.containerRef.current,
            this.itemRefs,
            this.props.initialVisibleIndex,
        );

        this.setState({
            scrollOffset,
            hasNext: getInitialHasNextByOffset(
                this.containerRef.current,
                this.itemRefs,
                scrollOffset,
            ),
            hasPrev: getInitialHasPrevByOffset(
                this.containerRef.current,
                this.itemRefs,
                scrollOffset,
            ),
            containerHeight: this.containerRef.current
                ? this.containerRef.current.getBoundingClientRect().height
                : this.state.containerHeight,
            lazyLoadLimit: this.props.lazyLoad
                ? Math.max(1, this.getLazyLoadingLimit())
                : 0,
        });
    }

    componentDidUpdate(prevProps: ICarouselProps): void {
        const {onLastItemVisible, children} = this.props;
        const prevLength = Children.count(prevProps.children);
        const length = Children.count(children);

        if (onLastItemVisible && length !== prevLength) {
            this.handleLastItemVisible = _once(onLastItemVisible);
        }

        if (children !== prevProps.children) {
            this.checkScrollBounds();
        }
    }

    private checkScrollBounds = (): void => {
        const {leftOffset, rightOffset = 0, scroll} = this.props;
        const {scrollOffset, hasPrev, hasNext} = this.state;
        const list = this.listRef.current;

        if (!list || scroll === 'native') {
            return;
        }

        const leftmostOffset = getPrevOffsetByRefs(
            list,
            this.itemRefs[0],
            leftOffset,
        );
        const rightmostOffset = Math.max(
            leftmostOffset,
            Math.min(
                list.scrollWidth - list.clientWidth + rightOffset,
                getNextOffsetByRefs(
                    list,
                    this.itemRefs[this.itemRefs.length - 1],
                    rightOffset,
                ),
            ),
        );

        let boundScrollOffset = Math.max(
            leftmostOffset,
            Math.min(rightmostOffset, scrollOffset),
        );
        let newHasPrev = boundScrollOffset > leftmostOffset;
        let newHasNext = boundScrollOffset < rightmostOffset - rightOffset;

        if (list.scrollWidth === list.clientWidth) {
            boundScrollOffset = 0;
            newHasPrev = false;
            newHasNext = false;
        }

        if (
            boundScrollOffset !== scrollOffset ||
            newHasNext !== hasNext ||
            newHasPrev !== hasPrev
        ) {
            this.setState({
                scrollOffset: boundScrollOffset,
                hasNext: newHasNext,
                hasPrev: newHasPrev,
            });
        }
    };

    /**
     * Проверяет наличие невидимых элементов, и при наличии отображает кнопку
     */
    checkNextItems = (): void => {
        const index = this.getInvisibleRightIndex();

        this.setState({
            hasNext: index !== undefined,
        });
    };

    /**
     * Прокручивает карусель так, чтобы элемент по индексу, встал у левого края с leftOffset
     * @comment Реализовано только для нативного скролла
     */
    setLeftItem = (index: number): void => {
        const {scroll, leftOffset} = this.props;

        if (scroll === 'native' && this.containerRef.current) {
            const nextPosition = getPrevOffsetByRefs(
                this.listRef.current,
                this.itemRefs[index],
                leftOffset,
            );

            this.containerRef.current.scrollTo({
                left: nextPosition,
                behavior: 'smooth',
            });
        }
    };

    /**
     * Прокручивает карусель так, чтобы элемент по индексу, встал у правого края с rightOffset
     * @comment Реализовано только для нативного скролла
     */
    setRightItem = (index: number): void => {
        const {scroll, rightOffset} = this.props;

        if (scroll === 'native' && this.containerRef.current) {
            const nextPosition = getNextOffsetByRefs(
                this.listRef.current,
                this.itemRefs[index],
                rightOffset,
            );

            this.containerRef.current.scrollTo({
                left: nextPosition,
                behavior: 'smooth',
            });
        }
    };

    /**
     * Прокручивает карусель так, чтобы элемент по индексу, встал в центре карусели
     * @comment Реализовано только для нативного скролла
     */
    setItemCenter = (index: number, options?: ScrollIntoViewOptions): void => {
        const {scroll} = this.props;
        const carouselItem = this.itemRefs[index];

        if (scroll === 'native' && carouselItem) {
            carouselItem.scrollIntoView(
                options || DEFAULT_SCROLL_CENTER_OPTIONS,
            );
        }
    };

    /**
     * @comment Ie скроллит без анимации, из-за поддержки методов scroll, scrollBy, scrollTo
     */
    scrollBy = (offset: number): void => {
        const {scroll} = this.props;

        if (scroll === 'native' && this.containerRef.current) {
            if (this.containerRef.current.scrollBy) {
                this.containerRef.current.scrollBy({
                    left: offset,
                    behavior: 'smooth',
                });
            } else {
                const scrollOffset =
                    this.containerRef.current.scrollLeft + offset;

                this.containerRef.current.scrollLeft = scrollOffset;
            }
        }
    };

    /**
     * Возвращает ширину элемента
     */
    getItemWidth(
        index: number,
        addSpaceBetween: boolean = false,
    ): number | undefined {
        if (!this.listRef.current) {
            return undefined;
        }

        const child = this.listRef.current.children[index] as HTMLDivElement;

        if (!child) {
            return undefined;
        }

        const add = addSpaceBetween ? this.getSpaceBetweenValue() : 0;

        return child.offsetWidth + add;
    }

    /* Helper */

    private getSpaceBetweenValue(): number {
        const {spaceBetween = 0} = this.props;

        return BASE_SPACE * (spaceBetween as number);
    }

    private getInvisibleRightIndex(): number | undefined {
        return getInvisibleRightIndex(this.containerRef.current, this.itemRefs);
    }

    private getLazyLoadingLimit(): number {
        const {children, type, lazyLoadSizes} = this.props;
        const childrenLength = Children.count(children);

        if (type === 'mini') {
            return 2;
        }

        if (lazyLoadSizes && this.containerRef.current) {
            const index = getInvisibleRightIndexBySizes(
                this.containerRef.current,
                lazyLoadSizes,
            );

            if (index) {
                return index;
            }
        } else {
            const index = this.getInvisibleRightIndex() || childrenLength - 2;

            return index + 1;
        }

        return childrenLength - 1;
    }

    private getLazyItemWidth(index: number): number | string | undefined {
        const {lazyLoadSizes, lazyLoad, type, getItemWidth} = this.props;
        const {containerHeight} = this.state;
        const size = lazyLoadSizes && lazyLoadSizes[index];

        if (getItemWidth) {
            const itemWidth = getItemWidth(index);

            if (itemWidth) {
                return itemWidth;
            }
        }

        if (size && lazyLoad && type !== 'mini') {
            const aspectRatio = containerHeight / size.height;

            return size.width * aspectRatio;
        }

        return undefined;
    }

    /* Handlers */

    private handlePrevClick = (): void => {
        const {leftOffset, onPrevButtonClick} = this.props;

        const index =
            getInvisibleLeftIndex(this.containerRef.current, this.itemRefs) ||
            0;

        const nextPosition = getPrevOffsetByRefs(
            this.listRef.current,
            this.itemRefs[index],
            leftOffset,
        );

        this.setState({
            scrollOffset: nextPosition,
            hasNext: true,
            hasPrev: index !== 0,
        });

        if (onPrevButtonClick) {
            onPrevButtonClick();
        }
    };

    private handleNextClick = (): void => {
        const {rightOffset, children, onNextButtonClick} = this.props;
        const childrenLength = Children.count(children);
        const index = this.getInvisibleRightIndex() ?? childrenLength - 1;

        const nextPosition = getNextOffsetByRefs(
            this.listRef.current,
            this.itemRefs[index],
            rightOffset,
        );

        this.setState({
            scrollOffset: nextPosition,
            hasNext: index + 1 !== childrenLength,
            hasPrev: true,
            lazyLoadLimit: this.state.lazyLoadLimit + 1,
        });

        if (onNextButtonClick) {
            onNextButtonClick();
        }
    };

    private handleWheel = (ev: React.WheelEvent): void => {
        const {scroll} = this.props;

        if (scroll === 'native' && this.containerRef.current) {
            if (Math.abs(ev.deltaX) < Math.abs(ev.deltaY)) {
                this.containerRef.current.scrollBy(ev.deltaY, 0);
            }
        }
    };

    /* Render */

    private renderItem = (item: ReactNode, index: number): ReactNode => {
        const {itemClassName, lazyLoad, lazyPlaceholder} = this.props;
        const {lazyLoadLimit} = this.state;

        const width = this.getLazyItemWidth(index);

        const canRenderItem = index < lazyLoadLimit || !lazyLoad;

        return (
            <div
                className={classNames(cx('carousel__item'), itemClassName)}
                ref={(instance): void => {
                    this.itemRefs[index] = instance;
                }}
                style={{width}}
                key={index}
            >
                {canRenderItem ? item : lazyPlaceholder}
            </div>
        );
    };

    render(): ReactNode {
        const {scrollOffset, hasNext, hasPrev} = this.state;
        const {
            children,
            type,
            height,
            arrowsType,
            arrowsPosition,
            className,
            listClassName,
            buttonClassName,
            gradient,
            spaceBetween,
            scroll,
            hideArrows,
            spaceBefore,
            spaceAfter,
            hideDisabledArrows,
        } = this.props;

        const classes = cx(
            'carousel',
            {carousel_type_full: type === 'mini'},
            {carousel_scroll_native: scroll === 'native'},
            {carousel_gradient: gradient},
            {carousel_gradient_prev: gradient && hasPrev},
            {carousel_gradient_next: gradient && hasNext},
            {[`carousel_withBefore_${spaceBefore}`]: spaceBefore},
            {[`carousel_withAfter_${spaceAfter}`]: spaceAfter},
        );

        const style = {
            transform: `translateX(${-scrollOffset}px)`,
            height: height,
        };

        const canShowArrows = !hideArrows && scroll === 'arrow';
        const hidePrevArrow = hideDisabledArrows && !hasPrev;
        const hideNextArrow = hideDisabledArrows && !hasNext;

        return (
            <div
                className={classNames(classes, className)}
                onScroll={this.handleScroll}
                onWheel={this.handleWheel}
                ref={this.containerRef}
            >
                {canShowArrows && !hidePrevArrow && (
                    <CarouselButton
                        type="prev"
                        arrowsType={arrowsType}
                        arrowsPosition={arrowsPosition}
                        onClick={this.handlePrevClick}
                        disabled={!hasPrev}
                        className={buttonClassName}
                    />
                )}
                <Box
                    between={spaceBetween}
                    className={cx('carousel__list', listClassName)}
                    style={style}
                    innerRef={this.listRef}
                    inline
                >
                    {Children.map(children, this.renderItem)}
                </Box>
                {canShowArrows && !hideNextArrow && (
                    <CarouselButton
                        type="next"
                        arrowsType={arrowsType}
                        arrowsPosition={arrowsPosition}
                        onClick={this.handleNextClick}
                        disabled={!hasNext}
                        className={buttonClassName}
                    />
                )}
            </div>
        );
    }
}

export default Carousel;
