import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {usePrevious} from 'utilities/hooks/usePrevious';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import BorderPositioned, {
    TDirection,
} from 'components/BorderPositioned/BorderPositioned';

import cx from './CornerCounter.scss';

interface ICornerProps extends IWithQaAttributes {
    count: number | string;
    renderZero?: boolean;
    direction?: TDirection;
}

interface ICornerCounterProps
    extends IWithClassName,
        ICornerProps,
        IWithQaAttributes {}

const Corner: React.FC<ICornerProps> = ({count, renderZero}) => {
    const previousCount = usePrevious<ICornerCounterProps['count']>(count);
    const shown = Boolean(Number(count) || renderZero);
    const countToShow = shown ? count : previousCount;

    return (
        <div className={cx('corner', {corner_shown: shown})}>
            <span className={cx('counter')}>{countToShow}</span>
        </div>
    );
};

const CornerCounter: React.FC<ICornerCounterProps> = ({
    className,
    children,
    count,
    renderZero = false,
    direction = 'top-right',
    ...rest
}) => {
    const corner = useMemo(
        () => <Corner count={count} renderZero={renderZero} />,
        [count, renderZero],
    );

    return (
        <BorderPositioned
            className={className}
            direction={direction}
            badge={corner}
            {...prepareQaAttributes(rest)}
        >
            {children}
        </BorderPositioned>
    );
};

export default CornerCounter;
