import React from 'react';

import {IWithClassName} from 'types/withClassName';
import EAsyncStatus from 'types/common/EAsyncStatus';
import ICountryRestrictionsInfo from 'components/CountryRestrictions/types/ICountryRestrictionsInfo';
import {
    ECountryRestrictionsColor,
    IGetExtendedCountryRestrictionsResponse,
} from 'server/api/AviaCountryRestrictionsApi/types/TAviaCountryRestrictionsApiResponse';

import {needShowCountryRestrictions} from 'projects/avia/utilities/needShowCountryRestrictions';
import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nBlock from 'i18n/covidBanner';

import {TCardRadius} from 'components/Card/Card';
import CountryRestrictionsSkeleton from 'components/CountryRestrictions/components/CountryRestrictionsSkeleton/CountryRestrictionsSkeleton';
import CountryRestrictionsPreview from 'components/CountryRestrictions/components/CountryRestrictionsPreview/CountryRestrictionsPreview';
import CountryRestrictionsDetailed from './components/CountryRestrictionsDetailed/CountryRestrictionsDetailed';

export interface ICountryRestrictionsProps extends IWithClassName {
    /** Ответ ручки avia_country_restrictions */
    countryRestrictions: ICountryRestrictionsInfo;

    forceSkeletons?: boolean;
    /** Название страны из пункта "откуда" на русском языке */
    fromCountryTitle: string;
    /** Название страны из пункта "куда" на русском языке */
    toCountryTitle: string;
    /** Скругление углов у всего блока */
    borderRadius?: TCardRadius;
}

const CountryRestrictions: React.FC<ICountryRestrictionsProps> = ({
    className,
    countryRestrictions,
    forceSkeletons = false,
    fromCountryTitle,
    toCountryTitle,
    borderRadius,
}) => {
    const {value: isOpened, setFalse: close, toggle} = useBoolean(false);

    const toggleOpensModal =
        Boolean(countryRestrictions.data?.expandedBannerMetrics?.length) &&
        countryRestrictions.data?.color !== ECountryRestrictionsColor.RED;

    const noInfoData: IGetExtendedCountryRestrictionsResponse = {
        color: ECountryRestrictionsColor.GRAY,
        titleText: toCountryTitle,
        desktopMinimized: [i18nBlock.noInfoAboutTourism()],
        mobileMinimized: [i18nBlock.noInfoAboutTourism()],
        expandedBannerMetrics: [],
    };

    if (!needShowCountryRestrictions(fromCountryTitle, toCountryTitle)) {
        return null;
    }

    if (forceSkeletons || countryRestrictions.status === EAsyncStatus.LOADING) {
        return <CountryRestrictionsSkeleton className={className} />;
    }

    if (countryRestrictions.status === EAsyncStatus.ERROR) {
        return null;
    }

    return (
        <>
            <CountryRestrictionsPreview
                className={className}
                borderRadius={borderRadius}
                countryRestrictions={countryRestrictions.data || noInfoData}
                toggleOpensModal={toggleOpensModal}
                isOpened={isOpened}
                toggle={toggle}
            />
            {countryRestrictions.data && toggleOpensModal && (
                <CountryRestrictionsDetailed
                    countryRestrictions={countryRestrictions.data}
                    isOpened={isOpened}
                    close={close}
                />
            )}
        </>
    );
};

export default CountryRestrictions;
