import React, {useMemo} from 'react';

import {IGetExtendedCountryRestrictionsResponse} from 'server/api/AviaCountryRestrictionsApi/types/TAviaCountryRestrictionsApiResponse';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/covidBanner';

import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import Link from 'components/Link/Link';
import DescriptionRestriction from 'components/CountryRestrictions/components/CountryRestrictionsDetailed/DescriptionRestriction';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import Box from 'components/Box/Box';

import cx from './CountryRestrictionsDetailed.scss';

interface ICountryRestrictionsDetailedProps {
    countryRestrictions: IGetExtendedCountryRestrictionsResponse;
    isOpened: boolean;
    close: () => void;
    footer?: React.ReactNode;
    loading?: boolean;
}

const CountryRestrictionsDetailed: React.FC<ICountryRestrictionsDetailedProps> =
    ({countryRestrictions, isOpened, close, footer, loading}) => {
        const deviceType = useDeviceType();
        const {isMobile} = deviceType;
        const {titleText} = countryRestrictions;

        const footerElement = useMemo(() => {
            if (footer) return footer;

            return (
                <>
                    <div>
                        {i18nBlock.foundInfoMistakeEnlarged()}&nbsp;
                        <Link
                            url={i18nBlock.informAboutMistakeFormLink()}
                            target="_blank"
                        >
                            {i18nBlock.informAboutMistakeLinkText()}
                        </Link>
                    </div>

                    <div className={cx('warningText')}>
                        {i18nBlock.warningText()}
                    </div>
                </>
            );
        }, [footer]);

        return (
            <ModalOrBottomSheet isVisible={isOpened} onClose={close}>
                <div className={cx('root', deviceMods('root', deviceType))}>
                    {loading ? (
                        <Box between={4}>
                            <TextSkeleton
                                withAnimation
                                size={isMobile ? 'l' : 'xl'}
                            />
                            <TextSkeleton
                                withAnimation
                                size={isMobile ? 'l' : 'xl'}
                            />
                            <TextSkeleton
                                withAnimation
                                size={isMobile ? 'l' : 'xl'}
                            />
                        </Box>
                    ) : (
                        <>
                            <Text
                                size={isMobile ? 'l' : 'xl'}
                                weight="bold"
                                tag="div"
                            >
                                {titleText}
                            </Text>

                            <div className={cx('actualForRussia')}>
                                {i18nBlock.actualForRussia()}
                            </div>

                            <DescriptionRestriction
                                countryRestrictions={countryRestrictions}
                            />
                        </>
                    )}

                    <Separator className={cx('separator')} />

                    {footerElement}
                </div>
            </ModalOrBottomSheet>
        );
    };

export default CountryRestrictionsDetailed;
