import React from 'react';
import {cloneDeep} from 'lodash';

import {
    ERichStringBlockType,
    IRichString,
} from 'server/api/AviaCountryRestrictionsApi/types/IRichString';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Text, {ITextProps} from 'components/Text/Text';
import Link from 'components/Link/Link';

export interface IRichStringProps {
    text: IRichString | undefined;
    /**
     * Если конкатенированная строка не содержит в конце данной подстроки, то
     * она будет добавлена в виде текста.
     */
    endOfString?: string;
    /**
     * Строка с теми символами, которые нужно удалить с конца строки.
     */
    deleteSymbolsFromEndOfString?: string;
    textProps?: ITextProps;
}

/**
 * Конкатенирует строку из массива простых строк и урлов
 */
const RichString: React.FC<IRichStringProps> = ({
    text,
    endOfString,
    deleteSymbolsFromEndOfString,
    textProps,
}) => {
    const {isMobile} = useDeviceType();

    if (text === undefined || text.data.length === 0) {
        return null;
    }

    const textArr = [...text.data];
    const lastIndex = textArr.length - 1;
    let lastElement = textArr[lastIndex];

    if (deleteSymbolsFromEndOfString) {
        const deleteRegExp = new RegExp(
            `[${deleteSymbolsFromEndOfString
                .split('')
                .map(char => `\\${char}`)}]$`,
        );

        if (deleteRegExp.test(lastElement.data.text)) {
            lastElement = textArr[lastIndex] = cloneDeep(lastElement);
            lastElement.data.text = lastElement.data.text.replace(
                deleteRegExp,
                '',
            );
        }
    }

    if (endOfString && !lastElement.data.text.endsWith(endOfString)) {
        textArr.push({
            blockType: ERichStringBlockType.TEXT,
            data: {
                text: endOfString,
            },
        });
    }

    return (
        <>
            {textArr.map((block, index) => {
                if (block.blockType === ERichStringBlockType.TEXT) {
                    return (
                        <Text
                            size={isMobile ? 's' : 'm'}
                            key={index}
                            {...textProps}
                        >
                            {block.data.text}
                        </Text>
                    );
                }

                return (
                    <Link url={block.data.url} target="_blank" key={index}>
                        {block.data.text}
                    </Link>
                );
            })}
        </>
    );
};

export default React.memo(RichString);
