import React, {useEffect, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {
    ECountryRestrictionsColor,
    IGetExtendedCountryRestrictionsResponse,
} from 'server/api/AviaCountryRestrictionsApi/types/TAviaCountryRestrictionsApiResponse';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';
import {reachGoalOnce} from 'utilities/metrika';
import {stopPropagationCallback} from 'utilities/dom/stopPropagationCallback';
import lowercaseFirstLetter from 'utilities/strings/lowercaseFirstLetter';
import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';
import {getBannerPreviewIcon} from 'components/CountryRestrictions/utilities/getBannerPreviewIcon';
import {isWordAbbreviation} from 'utilities/strings/isFirstWordAbbreviation';

import * as i18nBlock from 'i18n/covidBanner';

import Card, {TCardRadius} from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import Intersperse from 'components/Intersperse/Intersperse';
import Text from 'components/Text/Text';
import Box from 'components/Box/Box';
import Link from 'components/Link/Link';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import ArrowRightIconDesktop from 'icons/24/ArrowRight';
import ArrowRightIconMobile from 'icons/16/ArrowRight';

import cx from './CountryRestrictionsPreview.scss';

interface ICountryRestrictionsPreviewProps extends IWithClassName {
    borderRadius?: TCardRadius;
    countryRestrictions: IGetExtendedCountryRestrictionsResponse;
    toggleOpensModal: boolean;
    isOpened: boolean;
    toggle: () => void;
}

const isFirstWordAbbreviation = (str: string): boolean => {
    return isWordAbbreviation(str.split(' ')[0]);
};

const CountryRestrictionsPreview: React.FC<ICountryRestrictionsPreviewProps> =
    ({
        className,
        borderRadius,
        countryRestrictions,
        toggleOpensModal,
        isOpened,
        toggle,
    }) => {
        const deviceType = useDeviceType();
        const {isMobile} = deviceType;
        const {color, mobileMinimized, desktopMinimized} = countryRestrictions;

        const IconComponent = useMemo(
            () => getBannerPreviewIcon(color, deviceType),
            [color, deviceType],
        );
        const messages = useMemo(
            () => (isMobile ? mobileMinimized : desktopMinimized),
            [desktopMinimized, isMobile, mobileMinimized],
        );

        const arrowNode = useMemo(() => {
            if (!toggleOpensModal) {
                return <RotatingArrowIcon rotated={isOpened} />;
            }

            if (isMobile) {
                return <ArrowRightIconMobile />;
            }

            return <ArrowRightIconDesktop />;
        }, [isMobile, isOpened, toggleOpensModal]);

        useEffect(() => {
            reachGoalOnce(EAviaGoal.SEARCH_COVID_BANNER, {
                avia: {
                    covidBannerColor:
                        color === ECountryRestrictionsColor.GRAY
                            ? 'grey'
                            : color,
                },
            });
            // Пустой массив зависимостей для того, чтобы отправлять событие только один раз
        }, []);

        return (
            <Card
                className={cx(
                    'root',
                    deviceModMobile('root', deviceType),
                    `background_${countryRestrictions.color}`,
                    className,
                )}
                x={isMobile ? 4 : 8}
                y={4}
                onClick={toggle}
                radius={borderRadius}
            >
                <Flex
                    inline
                    between={4}
                    alignItems="center"
                    justifyContent="space-between"
                    flexWrap="nowrap"
                >
                    <IconComponent className={cx('icon')} />

                    <Box className={cx('message_block')}>
                        {isMobile ? (
                            <Text className={cx('title')}>
                                {countryRestrictions.titleText}
                            </Text>
                        ) : (
                            <Heading level={3} weight="bold">
                                {countryRestrictions.titleText}
                            </Heading>
                        )}

                        <div className={cx('textContainer')}>
                            <Intersperse
                                separator={<span className={cx('separator')} />}
                            >
                                {messages &&
                                    messages.map((message, index) => (
                                        <span key={index}>
                                            {index &&
                                            !isFirstWordAbbreviation(message)
                                                ? lowercaseFirstLetter(message)
                                                : capitalizeFirstLetter(
                                                      message,
                                                  )}
                                        </span>
                                    ))}
                            </Intersperse>
                        </div>
                    </Box>
                    <div className={cx('arrow')}>{arrowNode}</div>
                </Flex>

                {!toggleOpensModal && (
                    <AutoHeightAnimateControl canShowComponent={isOpened}>
                        <Text size="s-inset" className={cx('extra_info_text')}>
                            {i18nBlock.moreAboutChanges1()}
                            <Link
                                url={i18nBlock.moreAboutChangesUrl()}
                                onClick={stopPropagationCallback}
                                target="_blank"
                            >
                                {i18nBlock.moreAboutChanges2()}
                            </Link>
                        </Text>
                        <Text size="s-inset" className={cx('extra_info_text')}>
                            {i18nBlock.haveInformationAboutConditionsOfEntry()}
                            <Link
                                url={i18nBlock.informAboutMistakeFormLink()}
                                onClick={stopPropagationCallback}
                                target="_blank"
                            >
                                {i18nBlock.informAboutMistakeLinkText()}
                            </Link>
                        </Text>
                    </AutoHeightAnimateControl>
                )}
            </Card>
        );
    };

export default CountryRestrictionsPreview;
