import React from 'react';
import chunk from 'lodash/chunk';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/common-crossLinks';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';

import cx from './CrossLinks.scss';

export interface ICrossLinkItem {
    text: string;
    url: string;
}

const ONE_COLUMN_THRESHOLD = 3;

function ceil<T>(items: T[]): T[][] {
    return chunk(items, Math.ceil(items.length / 2));
}

function shift<T>(items: T[]): T[][] {
    if (items.length < ONE_COLUMN_THRESHOLD) {
        return [items];
    }

    const firstColumnLastIndex = Math.ceil((items.length + 1) / 2);

    return [
        [...items.slice(0, firstColumnLastIndex)],
        [...items.slice(firstColumnLastIndex, items.length)],
    ];
}

const columnAlgorithms = {
    ceil,
    shift,
};

interface ICrossLinksProps extends IWithClassName {
    title?: string;
    items: ICrossLinkItem[];
    columnsAlgorithm?: keyof typeof columnAlgorithms;
    onLinkClick?: () => void;
    columnClassName?: string;
}

const CrossLinks: React.FC<ICrossLinksProps> = props => {
    const {
        title = i18nBlock.title(),
        items,
        className,
        columnsAlgorithm = 'ceil',
        columnClassName,
        onLinkClick,
    } = props;

    const {isMobile} = useDeviceType();

    const makeColumns = columnAlgorithms[columnsAlgorithm];
    const columns = isMobile ? [items] : makeColumns(items);

    return (
        <Box className={className} between={4}>
            <Text size="xl" weight="bold" tag="div">
                {title}
            </Text>

            <Flex>
                {columns.map((columnItems, index) => (
                    <Flex
                        className={cx('column', columnClassName)}
                        key={index}
                        flexDirection="column"
                        between={3}
                    >
                        {columnItems.map(({text, url}) => (
                            <Link
                                className={cx('link')}
                                key={url}
                                to={{
                                    pathname: url,
                                    state: {
                                        scrollTop: 0,
                                    },
                                }}
                                onClick={onLinkClick}
                            >
                                {text}
                            </Link>
                        ))}
                    </Flex>
                ))}
            </Flex>
        </Box>
    );
};

export default React.memo(CrossLinks);
