import React, {useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EOrganicGoals} from 'utilities/metrika/types/goals/organic';
import {TOrganicSource} from 'utilities/metrika/types/params/organic';
import {SearchPageLayoutView} from 'types/hotels/search/ISearchLayout';
import {ICrossSaleHotels} from 'types/common/seoPages/ICrossSaleHotelsBlock';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getRegionUrl} from 'projects/hotels/utilities/urls/getRegionUrl';
import {getSearchUrlWithQuery} from 'projects/hotels/utilities/urls/getSearchUrlWithQuery';

import Box from 'components/Box/Box';
import Price from 'components/Price/Price';
import Heading from 'components/Heading/Heading';
import CrossSaleMap from 'components/CrossSaleMap/CrossSaleMap';

interface ICrossSaleHotelsBlockProps extends IWithClassName, ICrossSaleHotels {
    linkType: 'region' | 'search';
    vertical: TOrganicSource;
}

const CrossSaleHotelsBlock = React.forwardRef<
    HTMLDivElement,
    ICrossSaleHotelsBlockProps
>(function CrossSaleHotels(props, ref) {
    const {className, title, vertical, linkType, ...mapProps} = props;
    const {minPriceInRegion, region} = mapProps;
    const deviceType = useDeviceType();

    const handleClick = useCallback(() => {
        reachGoal(EOrganicGoals.CROSS_SALES_SEO, {
            organic: {
                crossSale: {
                    hotels: vertical,
                },
            },
        });
    }, [vertical]);

    const linkUrl = useMemo(() => {
        switch (linkType) {
            case 'region':
                return getRegionUrl({regionSlug: region.slug || ''});

            case 'search':
                return getSearchUrlWithQuery({
                    geoId: region.geoId,
                    serpView: SearchPageLayoutView.LIST,
                });
        }
    }, [linkType, region.slug, region.geoId]);

    return (
        <Box className={className} ref={ref}>
            {title && (
                <Box below={deviceType.isDesktop ? 5 : 4}>
                    <Heading level={2}>{title}</Heading>
                </Box>
            )}

            <CrossSaleMap
                crossSaleData={mapProps}
                linkUrl={linkUrl}
                showMapOnIntersection
                onLinkClick={handleClick}
                additionalTitle={
                    minPriceInRegion ? (
                        <Price {...minPriceInRegion} isFrom />
                    ) : null
                }
            />
        </Box>
    );
});

export default CrossSaleHotelsBlock;
