import {ReactNode, forwardRef} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IHotelsCrossSale} from 'server/api/HotelsCrossSaleApi/types/IHotelsCrossSale';
import IGetCrossSaleHotelsResponseV1 from 'server/api/HotelsCrossSaleApi/types/IGetCrossSaleHotelsResponseV1';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as keyset from 'i18n/avia-crossSale';

import HotelIcon from 'icons/16/Hotel';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Card, {TCardShadow} from 'components/Card/Card';
import Link from 'components/Link/Link';
import HotelsCrossSaleMap from 'components/HotelsCrossSaleMap/HotelsCrossSaleMap';
import IconWithRoundBackground from 'components/IconWithRoundBackground/IconWithRoundBackground';

import cx from './CrossSaleMap.scss';

export interface ICrossSaleMap extends IWithClassName, IWithQaAttributes {
    crossSaleData: IGetCrossSaleHotelsResponseV1 | IHotelsCrossSale;
    linkUrl: string;
    additionalTitle?: ReactNode;
    showMapOnIntersection?: boolean;
    shadow?: TCardShadow;
    onLinkClick?(): void;
}

const CrossSaleMap = forwardRef<HTMLElement, ICrossSaleMap>((props, ref) => {
    const {
        className,
        crossSaleData,
        linkUrl,
        additionalTitle,
        showMapOnIntersection,
        shadow,
        onLinkClick,
    } = props;
    const deviceType = useDeviceType();
    const {isDesktop} = deviceType;
    const {region, totalHotelCount} = crossSaleData;

    return (
        <Card
            ref={ref}
            shadow={shadow ?? 'default'}
            x={isDesktop ? 8 : 0}
            y={isDesktop ? 4 : 0}
            className={cx('root', className, deviceMods('root', deviceType))}
            {...prepareQaAttributes(props)}
        >
            <Flex flexDirection="column">
                <Flex justifyContent="space-between" alignItems="center">
                    <Flex
                        alignItems={
                            isDesktop || !additionalTitle
                                ? 'center'
                                : 'flex-start'
                        }
                        className={cx('iconWithText')}
                    >
                        <IconWithRoundBackground
                            icon={<HotelIcon className={cx('hotelIcon')} />}
                            backgroundColor="label-secondary"
                            className={cx('icon')}
                        />

                        <Link
                            to={linkUrl}
                            target="_blank"
                            theme="black"
                            onClick={onLinkClick}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'link',
                            })}
                        >
                            <Text
                                size="m"
                                weight="bold"
                                className={cx('variantsText')}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'hotelsCount',
                                })}
                            >
                                {keyset.variantsCount({
                                    hotelsCount: totalHotelCount,
                                })}
                            </Text>

                            <Text
                                size="m"
                                weight="bold"
                                className={cx('variantsText')}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'city',
                                })}
                            >
                                {keyset.inCity({
                                    cityName:
                                        region.linguistics.prepositionalCase ||
                                        '',
                                })}
                            </Text>

                            {additionalTitle && (
                                <Text
                                    size="m"
                                    className={cx('dates')}
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: 'additionalTitle',
                                    })}
                                >
                                    {additionalTitle}
                                </Text>
                            )}
                        </Link>
                    </Flex>

                    {isDesktop && (
                        <Link
                            to={linkUrl}
                            className={cx('selectHotelLink')}
                            target="_blank"
                            onClick={onLinkClick}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'selectHotelLink',
                            })}
                        >
                            {keyset.selectHotel()}
                        </Link>
                    )}
                </Flex>
            </Flex>

            <HotelsCrossSaleMap
                className={cx('map')}
                linkUrl={linkUrl}
                crossSaleData={crossSaleData}
                showMapOnIntersection={showMapOnIntersection}
                handleLinkClick={onLinkClick}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'crossSaleMap',
                })}
            />
        </Card>
    );
});

CrossSaleMap.displayName = 'CrossSaleMap';

export default CrossSaleMap;
