import React, {RefObject, PureComponent, forwardRef, ReactNode} from 'react';

import {
    ICalendarPricesByType,
    TCalendarPrices,
} from 'types/common/calendarPrice/ICalendarPrice';
import {ECalendarType} from 'components/Calendar/types';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import isAfterDay from 'components/Calendar/utilities/isAfterDay';
import isSameDay from 'components/Calendar/utilities/isSameDay';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {humanFormatDate} from 'utilities/dateUtils';

import * as i18nBlock from 'i18n/components-DatePicker';

import WithHistoryBack from 'containers/withHistoryBack/WithHistoryBack';

import Popup from 'components/Popup/Popup';
import Calendar from 'components/Calendar/Calendar';
import Separator from 'components/Separator/Separator';
import LinkButton from 'components/LinkButton/LinkButton';
import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import DrawerContainer from 'components/DatePicker/DrawerContainer';
import {useNewCalendarEnabled} from 'components/DatePicker/hooks/useNewCalendarEnabled';
import DatePickerNew from 'components/DatePicker/DatePickerNew';
import DatePickerTrigger from './components/DatePickerTrigger/DatePickerTrigger';
import {TDatePickerTriggerValueFn} from 'components/DatePicker/components/DatePickerTrigger/DatePickerTriggerNew';
import DatePickerFooter, {
    IDatePickerFooterProps,
    IDatePickerFooterPropsByCalendarType,
} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';
import {ESearchControlPlaceholderPosition} from 'components/SearchControl/SearchControl';

import ScopeContext from 'contexts/ScopeContext';

import cx from './DatePicker.scss';

export type TDatePickerSize = 's' | 'm' | 'xl';

const POPUP_DIRECTIONS = [EPopupDirection.BOTTOM_LEFT];

export interface IDatePickerProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    isModalView: boolean;
    popModalHistoryOnUnmount?: boolean;
    minDate: Date;
    maxDate: Date;
    nowDate: Date;
    startDate: Nullable<Date>;
    startDateTrailingValue?: string;
    endDate: Nullable<Date>;
    triggerViewType: 'union' | 'tile';
    size: TDatePickerSize;
    hideDateClearButton?: boolean;
    triggerStartDateRef?: RefObject<HTMLElement>;
    triggerEndDateRef?: RefObject<HTMLElement>;
    triggerButtonName?: string;
    triggerWithThinBorder?: boolean;
    triggerClassName?: string;
    triggerFocusClassName?: string;
    canSelectSomeDays?: boolean;
    startDatePlaceholder: string;
    endDatePlaceholder?: string;
    canRenderResetIcon: boolean;
    resetRangeAfterFirstInteraction?: boolean;
    tabIndex?: number;
    headerNode?: React.ReactNode;
    footerBlockParams?: Omit<IDatePickerFooterProps, 'onButtonClick'>;
    footerBlockParamsByType?: IDatePickerFooterPropsByCalendarType;
    errorStartDate?: boolean | string[];
    errorEndDate?: boolean | string[];
    error?: boolean | string[];
    isShortTriggerFormatDate: boolean;
    triggerValue?: string | TDatePickerTriggerValueFn;
    triggerValueClassName?: string;
    canAutoHideCalendar?: boolean;
    canToggle?: boolean;
    cutPastWeeks?: boolean;
    prices?: TCalendarPrices;
    pricesByType?: ICalendarPricesByType;
    isRoundTrip?: boolean;
    datePickerPlaceholderPosition?: ESearchControlPlaceholderPosition;
    /**
     * Свойство по которому определяю, нужна ли дата обратно для отображения
     */
    hasEndDate?: boolean;
    onDaySelect: (params: {
        startDate: Nullable<Date>;
        endDate: Nullable<Date>;
        calendarType: ECalendarType;
    }) => void;
    onResetValue?: (calendarType: ECalendarType) => void;
    onFinishSelect?: (calendarType: ECalendarType) => void;
    onShowCalendar?: (calendarType: ECalendarType) => void;
    onHideCalendar?: (calendarType: ECalendarType) => void;
    onModalHistoryBack?: (calendarType: ECalendarType) => void;
    onFocus?: () => void;
    onBlur?: () => void;
}

interface IDatePickerState {
    visibleCalendarType: Nullable<ECalendarType>;
}

const MONTH_LIST_WIDTH = -128;
const TRIGGER_OFFSET = -16;

export class DatePicker extends PureComponent<
    IDatePickerProps,
    IDatePickerState
> {
    static readonly defaultProps: Partial<IDatePickerProps> = {
        canSelectSomeDays: true,
        isRoundTrip: false,
        hasEndDate: false,
        startDate: null,
        endDate: null,
        isModalView: false,
        startDatePlaceholder: '',
        endDatePlaceholder: '',
        canRenderResetIcon: true,
        headerNode: null,
        triggerViewType: 'union',
        size: 'xl',
        hideDateClearButton: false,
        errorStartDate: false,
        errorEndDate: false,
        isShortTriggerFormatDate: false,
        canAutoHideCalendar: true,
        cutPastWeeks: false,
    };

    private static getIsSwitcherMouseEvent(
        switcherRef: RefObject<DatePickerTrigger>,
        targetNode: HTMLElement,
    ): boolean | undefined {
        return switcherRef?.current?.labelRef?.current?.contains(targetNode);
    }

    readonly state: IDatePickerState = {
        visibleCalendarType: null,
    };

    private readonly calendarRef = React.createRef<typeof Calendar>();
    private readonly popupRef = React.createRef<HTMLDivElement>();

    private touchStartY: Nullable<number> = null;
    private hasSwipeUp = false;
    private drawerSwipeStarted = false;
    private scrollStartTop = 0;
    private focusedField: ECalendarType | null = null;

    // Нужен, чтобы шторка плавно скрывалась, а не просто схлопывалась при закрытии
    // на мобилке. При переходе на новую механику, можно будет спокойно удалить
    private lastShownCalendarType: Nullable<ECalendarType> =
        ECalendarType.StartDate;

    componentDidMount(): void {
        document.addEventListener('mousedown', this.handleOutsideInteraction);
        document.addEventListener('focus', this.handleOutsideInteraction, true);
    }

    componentDidUpdate(
        prevProps: Readonly<IDatePickerProps>,
        prevState: Readonly<IDatePickerState>,
    ): void {
        const {isRoundTrip} = this.props;

        this.lastShownCalendarType = prevState.visibleCalendarType;

        if (prevProps.isRoundTrip !== isRoundTrip) {
            const {visibleCalendarType} = this.state;

            if (visibleCalendarType === ECalendarType.EndDate) {
                this.handleFocus(ECalendarType.StartDate);
            }
        }
    }

    componentWillUnmount(): void {
        document.removeEventListener(
            'mousedown',
            this.handleOutsideInteraction,
        );
        document.removeEventListener(
            'focus',
            this.handleOutsideInteraction,
            true,
        );
    }

    private handleOutsideInteraction = (e: any) => {
        const {isModalView} = this.props;
        const {visibleCalendarType} = this.state;

        // eslint-disable-next-line no-eq-null
        if (visibleCalendarType != null && !isModalView) {
            const targetNode = e.target;

            const isStartSwitcherMouseEvent =
                DatePicker.getIsSwitcherMouseEvent(
                    this.startDateTriggerRef,
                    targetNode,
                );
            const isEndSwitcherMouseEvent = DatePicker.getIsSwitcherMouseEvent(
                this.endDateTriggerRef,
                targetNode,
            );
            const isPopupMouseEvent =
                this.popupRef?.current?.contains(targetNode);

            if (
                !(
                    isStartSwitcherMouseEvent ||
                    isEndSwitcherMouseEvent ||
                    isPopupMouseEvent
                )
            ) {
                this.handleBlur();
            }
        }
    };

    /* Refs manipulate */

    readonly startDateTriggerRef = React.createRef<DatePickerTrigger>();
    readonly endDateTriggerRef = React.createRef<DatePickerTrigger>();

    private getDateTriggerRefByType(
        calendarType: ECalendarType,
    ): RefObject<DatePickerTrigger> {
        return calendarType === ECalendarType.StartDate
            ? this.startDateTriggerRef
            : this.endDateTriggerRef;
    }

    private handlers: {[calendarType in ECalendarType]?: () => void} = {};

    private getPopupIsVisible(calendarType: Nullable<ECalendarType>): boolean {
        const {visibleCalendarType} = this.state;

        return visibleCalendarType === calendarType;
    }

    showPopup = (calendarType: ECalendarType): void => {
        this.setState({visibleCalendarType: calendarType}, () => {
            this.handleShowCalendar(calendarType);
        });
    };

    hidePopup = (): void => {
        const {visibleCalendarType} = this.state;

        if (!visibleCalendarType) {
            return;
        }

        this.setState({visibleCalendarType: null}, () => {
            this.handleHideCalendar(visibleCalendarType);
        });
    };

    private hideMobileDrawer = (): void => {
        const {visibleCalendarType} = this.state;

        this.hidePopup();

        const calendarType = visibleCalendarType || this.lastShownCalendarType;

        if (!calendarType) {
            return;
        }

        const ref = this.getDateTriggerRefByType(calendarType);

        // Блюр на тачах нужен, так как срабатывает событие onFocus
        // при переключении приложений и календарь переоткрывается
        ref.current?.blur();
    };

    /* Calendar handlers */
    private handleClickReset = (calendarType: ECalendarType): void => {
        const {onDaySelect, onResetValue, startDate, endDate} = this.props;

        if (calendarType === ECalendarType.StartDate) {
            onDaySelect({startDate: null, endDate, calendarType});
        } else {
            onDaySelect({startDate, endDate: null, calendarType});
        }

        if (onResetValue) {
            onResetValue(calendarType);
        }
    };

    private handleDaySelect = ({
        calendarType,
        selectedDate,
    }: {
        calendarType: ECalendarType;
        selectedDate: Nullable<Date>;
    }): void => {
        const {onDaySelect, canSelectSomeDays, startDate, endDate} = this.props;

        if (calendarType === ECalendarType.StartDate) {
            const isSameSelectedDateAndEndDate =
                selectedDate && endDate && isSameDay(selectedDate, endDate);
            const isSelectedDateAfterEndDate =
                endDate &&
                isAfterDay(selectedDate ?? Number(selectedDate), endDate);

            if (isSameSelectedDateAndEndDate || isSelectedDateAfterEndDate) {
                onDaySelect({
                    startDate: selectedDate,
                    endDate: null,
                    calendarType,
                });
            } else {
                onDaySelect({
                    startDate: selectedDate,
                    endDate,
                    calendarType,
                });
            }
        } else {
            const isSameSelectedDateAndStartDate =
                selectedDate && startDate && isSameDay(selectedDate, startDate);

            if (!canSelectSomeDays && isSameSelectedDateAndStartDate) {
                onDaySelect({
                    startDate: null,
                    endDate: selectedDate,
                    calendarType,
                });
            } else {
                onDaySelect({
                    startDate,
                    endDate: selectedDate,
                    calendarType,
                });
            }
        }

        this.checkFinishSelect(calendarType);
    };

    private handleFocus = (calendarType: ECalendarType): void => {
        const {onFocus} = this.props;
        const {visibleCalendarType} = this.state;
        const shouldPreventFocus = this.focusedField === calendarType;

        this.focusedField = calendarType;
        if (shouldPreventFocus) return;

        if (visibleCalendarType !== calendarType) {
            this.showPopup(calendarType);
        }

        onFocus?.();
    };

    private handleMouseDown = (calendarType: ECalendarType): void => {
        const {canToggle} = this.props;
        const {visibleCalendarType} = this.state;

        if (visibleCalendarType === calendarType) {
            if (canToggle) {
                this.hidePopup();
            }
        } else {
            this.showPopup(calendarType);
        }
    };

    private handleBlur = (): void => {
        const {onBlur} = this.props;

        this.focusedField = null;

        this.hidePopup();

        onBlur?.();
    };

    private handleShowCalendar = (calendarType: ECalendarType) => {
        const {onShowCalendar} = this.props;

        onShowCalendar?.(calendarType);
    };

    private handleHideCalendar = (calendarType: ECalendarType): void => {
        const {onHideCalendar} = this.props;

        onHideCalendar?.(calendarType);
    };

    private checkFinishSelect = (calendarType: ECalendarType): void => {
        const {onFinishSelect, isRoundTrip, canAutoHideCalendar, deviceType} =
            this.props;

        if (!canAutoHideCalendar) {
            return;
        }

        if (deviceType.isMobile) {
            this.handleBlur();

            return;
        }

        if (isRoundTrip && calendarType === ECalendarType.StartDate) {
            requestAnimationFrame(() => {
                const {endDate} = this.props;
                const {current} = this.endDateTriggerRef;

                if (!current) {
                    return;
                }

                if (endDate) {
                    this.handleBlur();
                } else {
                    current.focus();
                }
            });
        } else {
            onFinishSelect?.(calendarType);

            this.handleBlur();
        }
    };

    private getCommonTriggerPropsByType(calendarType: ECalendarType) {
        const {
            startDate,
            endDate,
            startDatePlaceholder,
            startDateTrailingValue,
            endDatePlaceholder,
            tabIndex,
            isRoundTrip,
            deviceType,
            triggerValue,
            triggerEndDateRef,
            triggerStartDateRef,
        } = this.props;

        const isStartDate = calendarType === ECalendarType.StartDate;

        const startDayTriggerValue =
            isStartDate && triggerValue
                ? triggerValue + (startDateTrailingValue || '')
                : undefined;

        let date = null;

        if (isStartDate) {
            date = startDate;
        } else if (isRoundTrip) {
            date = endDate;
        }

        return {
            deviceType,
            calendarType,
            placeholder: isStartDate
                ? startDatePlaceholder
                : endDatePlaceholder,
            date,
            triggerValue:
                isStartDate || isRoundTrip
                    ? startDayTriggerValue || triggerValue
                    : i18nBlock.endDateDotDontNeed(),
            triggerRef: isStartDate ? triggerStartDateRef : triggerEndDateRef,
            tabIndex,
            onClickReset: this.handleClickReset,
            ...prepareQaAttributes({
                parent: this.props,
                current: isStartDate ? 'start' : 'end',
            }),
        };
    }

    private getModalHistoryBackHandler(calendarType: Nullable<ECalendarType>) {
        if (!calendarType) {
            return undefined;
        }

        if (this.handlers[calendarType]) {
            return this.handlers[calendarType];
        }

        const {onModalHistoryBack} = this.props;

        if (!onModalHistoryBack) {
            return undefined;
        }

        this.handlers[calendarType] = onModalHistoryBack.bind(
            this,
            calendarType,
        );

        return undefined;
    }

    /* Modal View */

    private renderModalCalendar = (calendarType: ECalendarType): ReactNode => {
        const visible = this.getPopupIsVisible(calendarType);

        if (!visible) {
            return null;
        }

        return (
            <DrawerContainer className={cx('drawerContainer')}>
                {this.renderDrawerHeader(calendarType)}
                <Separator className={cx('separator')} />
                {this.renderCalendar(calendarType, false)}
            </DrawerContainer>
        );
    };

    private renderDrawerHeader = (calendarType: ECalendarType): ReactNode => {
        const {startDate, endDate, startDatePlaceholder, endDatePlaceholder} =
            this.props;

        const isStartDate = calendarType === ECalendarType.StartDate;
        const placeholder = isStartDate
            ? startDatePlaceholder
            : endDatePlaceholder;
        const startDateHuman = startDate
            ? ' — ' + humanFormatDate(startDate)
            : '';
        const endDateHuman = endDate ? ' — ' + humanFormatDate(endDate) : '';
        const date = isStartDate ? startDateHuman : endDateHuman;

        const onResetClick = (): void => {
            if (calendarType) {
                this.handleClickReset(calendarType);
            }
        };

        return (
            <div className={cx('drawerHeader')}>
                <div className={cx('text')}>
                    {placeholder}
                    {date}
                </div>
                <div>
                    <LinkButton
                        onClick={onResetClick}
                        className={cx('resetButton')}
                    >
                        {i18nBlock.reset()}
                    </LinkButton>
                </div>
            </div>
        );
    };

    /* Dropdown View */

    private renderCalendarTrigger = (
        calendarType: ECalendarType,
    ): ReactNode => {
        const {
            errorStartDate,
            errorEndDate,
            triggerClassName,
            triggerWithThinBorder,
            triggerButtonName,
            triggerViewType,
            triggerFocusClassName,
            isShortTriggerFormatDate,
            canRenderResetIcon,
            triggerValue,
            triggerValueClassName,
            deviceType: {isMobile},
            size,
            datePickerPlaceholderPosition,
        } = this.props;

        const isStartDate = calendarType === ECalendarType.StartDate;
        const visible = this.getPopupIsVisible(calendarType);
        const handleFocus = (): void => this.handleFocus(calendarType);
        const handleMouseDown = (): void => this.handleMouseDown(calendarType);

        return (
            <DatePickerTrigger
                className={cx(
                    triggerClassName,
                    'datePickerTrigger',
                    `datePickerTrigger_viewType_${triggerViewType}`,
                    {
                        datePickerTrigger_mobile: isMobile,
                        [`datePickerTrigger_type_${calendarType}`]:
                            calendarType,
                    },
                )}
                triggerValueClassName={triggerValueClassName}
                ref={this.getDateTriggerRefByType(calendarType)}
                // @ts-ignore костыль, но мы уверены, что сюда точно приходит строка, а не функция
                // при обновлении календаря уберем
                triggerValue={triggerValue}
                triggerButtonName={triggerButtonName}
                withThinBorder={triggerWithThinBorder}
                focusClassName={triggerFocusClassName}
                triggerViewType={triggerViewType}
                onMouseDown={handleMouseDown}
                onFocus={handleFocus}
                visiblePopup={visible}
                canRenderResetIcon={canRenderResetIcon}
                isShortFormatDate={isShortTriggerFormatDate}
                hasError={Boolean(isStartDate ? errorStartDate : errorEndDate)}
                size={size}
                placeholderPosition={datePickerPlaceholderPosition}
                {...this.getCommonTriggerPropsByType(calendarType)}
            />
        );
    };

    private onCalendarTouchStart = (e: TouchEvent): void => {
        if (!e.currentTarget) {
            return;
        }

        const {scrollTop} = e.currentTarget as HTMLElement;

        this.touchStartY = e.touches[0].clientY;
        this.scrollStartTop = scrollTop;
    };

    private onCalendarTouchMove = (e: TouchEvent): void => {
        const currentY = e.changedTouches[0].clientY;
        const startY = this.touchStartY as number;

        const swipeDown = startY < currentY;

        // Если календарь в самом начале и мы пытаемся проскролить еще вверх,
        // событие нужно отменить в браузерах с оверскролом
        if (e.cancelable && this.scrollStartTop === 0 && swipeDown) {
            e.preventDefault();

            // Считаем, что мы тянем шторку вниз только,
            // если пользователь за время события не свайпал вверх
            // Например, пользователь мог сначала прокрутить вверх, а потом чуть
            // прокрутить календарь вниз
            if (!this.hasSwipeUp) {
                this.drawerSwipeStarted = true;
            }
        }

        if (!swipeDown) {
            this.hasSwipeUp = true;
        }

        if (this.drawerSwipeStarted) {
            return;
        }

        // Если мы прокручиваем календарь, то событие не должно всплыть
        // в dragObserver шторки
        e.stopPropagation();
    };

    private onCalendarTouchEnd = (): void => {
        this.touchStartY = null;
        this.hasSwipeUp = false;
        this.drawerSwipeStarted = false;
    };

    private renderCalendarFooter = (calendarType: ECalendarType): ReactNode => {
        const {footerBlockParams, footerBlockParamsByType} = this.props;

        const footerProps =
            footerBlockParamsByType?.[calendarType] || footerBlockParams;

        if (!footerProps) {
            return null;
        }

        return (
            <div className={cx('footer')}>
                <DatePickerFooter {...footerProps} />
            </div>
        );
    };

    private renderCalendar = (
        calendarType: ECalendarType,
        isRenderMonthsList: boolean,
    ): ReactNode => {
        const {
            canSelectSomeDays,
            nowDate,
            minDate,
            maxDate,
            startDate,
            endDate,
            headerNode,
            cutPastWeeks,
            isModalView,
            prices,
            pricesByType,
        } = this.props;

        return (
            <Calendar
                // @ts-ignore typeof не справляется
                ref={this.calendarRef}
                className={cx(
                    isModalView ? 'drawerCalendar' : 'dropdownCalendar',
                )}
                calendarWrapperClassName={cx(
                    isModalView && 'drawerCalendarWrapper',
                )}
                monthsGridClassName={cx(
                    isModalView ? 'drawerMonthsGrid' : 'dropdownMonthsGrid',
                )}
                weekdaysClassName={cx(
                    isModalView ? 'drawerWeekdays' : 'dropdownWeekdays',
                )}
                minDate={minDate}
                maxDate={maxDate}
                nowDate={nowDate}
                endDate={endDate}
                startDate={startDate}
                headerNode={headerNode}
                onDaySelect={this.handleDaySelect}
                calendarType={calendarType}
                canSelectSomeDays={canSelectSomeDays}
                isRenderMonthsList={isRenderMonthsList}
                prices={prices || pricesByType?.[calendarType]}
                footerNode={this.renderCalendarFooter(calendarType)}
                cutPastWeeks={cutPastWeeks}
                onMonthsGridTouchStart={this.onCalendarTouchStart}
                onMonthsGridTouchEnd={this.onCalendarTouchEnd}
                onMonthsGridTouchMove={this.onCalendarTouchMove}
                {...prepareQaAttributes('calendar')}
            />
        );
    };

    private renderPopup(): ReactNode {
        const {visibleCalendarType} = this.state;

        const {isModalView, size} = this.props;

        const isDrawerVisible = Boolean(visibleCalendarType);

        const onModalHistoryBack =
            this.getModalHistoryBackHandler(visibleCalendarType);

        const lastCalendar = this.lastShownCalendarType as ECalendarType;

        if (isModalView) {
            return (
                <WithHistoryBack
                    isVisible={isDrawerVisible}
                    close={this.hidePopup}
                    onHistoryBack={onModalHistoryBack}
                >
                    <ModalOrBottomSheet
                        isVisible={isDrawerVisible}
                        modalProps={{
                            isMobile: true,
                        }}
                        bottomSheetProps={{
                            contentClassName: cx('bottomSheetContent'),
                            keepMounted: false,
                            padding: 0,
                        }}
                        onClose={this.hideMobileDrawer}
                    >
                        <div ref={this.popupRef}>
                            {this.renderModalCalendar(
                                visibleCalendarType || lastCalendar,
                            )}
                        </div>
                    </ModalOrBottomSheet>
                </WithHistoryBack>
            );
        }

        if (!visibleCalendarType) {
            return null;
        }

        const dateTriggerRef =
            this.getDateTriggerRefByType(visibleCalendarType);
        const anchor = dateTriggerRef?.current?.buttonRef?.current;

        if (!anchor) {
            return null;
        }

        return (
            <ScopeContext.Consumer>
                {(scope): ReactNode => (
                    <Popup
                        className={cx('popupContainer')}
                        visible
                        nonvisual={false}
                        anchor={{current: anchor}}
                        directions={POPUP_DIRECTIONS}
                        secondaryOffset={
                            size === 'xl'
                                ? MONTH_LIST_WIDTH + TRIGGER_OFFSET
                                : TRIGGER_OFFSET
                        }
                        mainOffset={size === 'xl' ? undefined : 8}
                        zIndex={105}
                        scope={scope ? {current: scope} : undefined}
                        plain
                    >
                        <div ref={this.popupRef}>
                            {this.renderCalendar(visibleCalendarType, true)}
                        </div>
                    </Popup>
                )}
            </ScopeContext.Consumer>
        );
    }

    render(): ReactNode {
        const {className, hasEndDate, size} = this.props;

        return (
            <div
                className={cx('datePicker', `datePicker_${size}`, className)}
                {...prepareQaAttributes(this.props)}
            >
                {this.renderCalendarTrigger(ECalendarType.StartDate)}
                {hasEndDate &&
                    this.renderCalendarTrigger(ECalendarType.EndDate)}
                {this.renderPopup()}
            </div>
        );
    }
}

const DatePickerSwitch = forwardRef<
    DatePicker | DatePickerNew,
    IDatePickerProps
>(function DatePickerSwitcher(props, ref) {
    const isNew = useNewCalendarEnabled();

    const Component = (isNew ? DatePickerNew : DatePicker) as React.ElementType;

    return (
        <Component
            canSelectRange={props.isRoundTrip}
            cutPastWeeks={props.cutPastWeeks || isNew}
            ref={ref}
            {...props}
        />
    );
});

export default DatePickerSwitch;
