import React, {
    ComponentType,
    MutableRefObject,
    PureComponent,
    ReactNode,
    RefObject,
} from 'react';

import {TCalendarPrices} from 'types/common/calendarPrice/ICalendarPrice';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {INullableDateRange} from 'utilities/dateUtils/types';
import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {EDatePickerSelectionMode} from 'components/DatePicker/types';
import {IIconProps} from 'icons/types/icon';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {formatDateRange} from 'utilities/dateUtils';
import sortDates from 'utilities/dateUtils/sortDates';

import * as i18nBlock from 'i18n/components-DatePicker';

import WithHistoryBack from 'containers/withHistoryBack/WithHistoryBack';

import Popup from 'components/Popup/Popup';
import Calendar from 'components/Calendar/CalendarNew';
import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import LinkButton from 'components/LinkButton/LinkButton';
import Separator from 'components/Separator/Separator';
import DrawerContainer from 'components/DatePicker/DrawerContainer';
import DatePickerFooter, {
    IDatePickerFooterProps,
    IDatePickerFooterPropsByCalendarType,
} from './components/DatePickerFooter/DatePickerFooter';
import ErrorTooltip from './components/ErrorTooltip/ErrorTooltip';
import DatePickerTrigger, {
    TDatePickerTriggerValueFn,
} from './components/DatePickerTrigger/DatePickerTriggerNew';

import ScopeContext from 'contexts/ScopeContext';

import cx from './DatePicker.scss';

export type TDatePickerSize = 's' | 'm' | 'xl';

const POPUP_DIRECTIONS = [EPopupDirection.BOTTOM_LEFT, EPopupDirection.BOTTOM];

export interface IModalContainerProps {
    triggerNode: React.ReactNode;
    componentNode: React.ReactNode;
    isFilled: boolean;
    hideModal: () => void;
}

export interface IDatePickerProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    isModalView: boolean;
    minDate: Date;
    maxDate: Date;
    nowDate: Date;
    startDate: Nullable<Date>;
    endDate: Nullable<Date>;
    triggerViewType: 'union' | 'tile';
    size: TDatePickerSize;
    hideDateClearButton?: boolean;
    // На некоторых вертикалях, например в авиа при каждом открытии календаря и последующем выборе
    // нужно сбрасывать интервал целиком, чтобы выбор даты "только туда" выглядел прозрачнее
    resetRangeAfterFirstInteraction?: boolean;
    triggerClassName?: string;
    triggerFocusClassName?: string;
    triggerValueClassName?: string;
    triggerPlaceholderClassName?: string;
    triggerIconClassName?: string;
    triggerIcon?: ComponentType<IIconProps>;
    placeholder?: string;
    canRenderResetIcon: boolean;
    tabIndex?: number;
    footerBlockParams?: Omit<IDatePickerFooterProps, 'onButtonClick'>;
    footerBlockParamsByType?: IDatePickerFooterPropsByCalendarType;
    headerNode?: React.ReactNode;
    error?: boolean | string[];
    isShortTriggerFormatDate: boolean;
    triggerValue?: string | TDatePickerTriggerValueFn;
    canAutoHideCalendar?: boolean;
    canToggle?: boolean;
    canSelectRange?: boolean;
    cutPastWeeks: boolean;
    prices?: TCalendarPrices;
    withYearSelect?: boolean;
    onDaySelect: (params: {
        startDate: Nullable<Date>;
        endDate: Nullable<Date>;
    }) => void;
    onResetValue?: () => void;
    onFinishSelect?: () => void;
    onShowCalendar?: () => void;
    onHideCalendar?: () => void;
    onModalHistoryBack?: () => void;
    onFocus?: () => void;
    onBlur?: () => void;
}

export interface IDatePickerState {
    isCalendarVisible: boolean;
    hoveredDate: Nullable<Date>;
    hasInteractionAfterShow: boolean;
}

const MONTH_LIST_WIDTH = -128;
const TRIGGER_OFFSET = -16;

class DatePickerNew extends PureComponent<IDatePickerProps, IDatePickerState> {
    static defaultProps = {
        hasEndDate: false,
        startDate: null,
        endDate: null,
        isModalView: false,
        startDatePlaceholder: '',
        endDatePlaceholder: '',
        canRenderResetIcon: true,
        triggerViewType: 'union',
        size: 'xl',
        hideDateClearButton: false,
        errorStartDate: false,
        errorEndDate: false,
        isShortTriggerFormatDate: false,
        canAutoHideCalendar: true,
        cutPastWeeks: false,
        placeholder: 'Когда',
        canSelectRange: false,
        headerNode: null,
    };

    private static getIsSwitcherMouseEvent(
        switcherRef: RefObject<DatePickerTrigger>,
        targetNode: HTMLElement,
    ): boolean | undefined {
        return switcherRef?.current?.labelRef?.current?.contains(targetNode);
    }

    state: IDatePickerState = {
        isCalendarVisible: false,
        hoveredDate: null,
        hasInteractionAfterShow: false,
    };

    private calendarRef = React.createRef<typeof Calendar>();
    private popupRef = React.createRef<HTMLDivElement>();
    private yearSelectRef: MutableRefObject<HTMLElement | null> = {
        current: null,
    };

    private touchStartY: Nullable<number> = null;
    private hasSwipeUp = false;
    private drawerSwipeStarted = false;
    private scrollStartTop = 0;

    componentDidMount(): void {
        document.addEventListener('mousedown', this.handleOutsideInteraction);
        document.addEventListener('focus', this.handleOutsideInteraction, true);
    }

    componentWillUnmount(): void {
        document.removeEventListener(
            'mousedown',
            this.handleOutsideInteraction,
        );
        document.removeEventListener(
            'focus',
            this.handleOutsideInteraction,
            true,
        );
    }

    private handleOutsideInteraction = (e: any): void => {
        const {isModalView} = this.props;
        const {isCalendarVisible} = this.state;

        if (isCalendarVisible && !isModalView) {
            const targetNode = e.target;

            const isSwitcherMouseEvent = DatePickerNew.getIsSwitcherMouseEvent(
                this.dateTriggerRef,
                targetNode,
            );
            const isPopupMouseEvent =
                this.popupRef?.current?.contains(targetNode);
            const isYearSelectEvent =
                this.yearSelectRef?.current?.contains(targetNode);

            if (
                !(
                    isSwitcherMouseEvent ||
                    isPopupMouseEvent ||
                    isYearSelectEvent
                )
            ) {
                this.handleBlur();
            }
        }
    };

    /* Refs manipulate */

    dateTriggerRef = React.createRef<DatePickerTrigger>();
    yearRef = React.createRef<DatePickerTrigger>();

    private backHandler?: () => void;

    showPopup = (): void => {
        this.setState(
            {isCalendarVisible: true, hasInteractionAfterShow: false},
            () => {
                this.handleShowCalendar();
            },
        );
    };

    hidePopup = (): void => {
        const {isCalendarVisible} = this.state;

        if (!isCalendarVisible) {
            return;
        }

        this.setState({isCalendarVisible: false}, () => {
            this.handleHideCalendar();
        });
    };

    private handleRangeUpdate = ({
        startDate,
        endDate,
    }: INullableDateRange): void => {
        const {onDaySelect} = this.props;

        onDaySelect({startDate, endDate});
        this.setState({hoveredDate: null, hasInteractionAfterShow: true});
    };

    hideMobileDrawer = (): void => {
        this.hidePopup();

        const ref = this.dateTriggerRef;

        // Блюр на тачах нужен, так как срабатывает событие onFocus
        // при переключении приложений и календарь переоткрывается
        ref.current?.blur();
    };

    /* Calendar handlers */
    private handleClickReset = (): void => {
        const {onResetValue} = this.props;

        this.handleRangeUpdate({startDate: null, endDate: null});

        if (onResetValue) {
            onResetValue();
        }
    };

    private handleDayHover = (date: Nullable<Date>): void => {
        this.setState({hoveredDate: date});
    };

    private handleDaySelect = ({
        selectedDate,
    }: {
        selectedDate: Nullable<Date>;
    }): void => {
        const {startDate} = this.props;
        const selectionMode = this.getSelectionMode();

        let newRange: INullableDateRange = {
            startDate: null,
            endDate: null,
        };

        if (selectionMode === EDatePickerSelectionMode.START_DATE) {
            newRange = {
                startDate: selectedDate,
                endDate: null,
            };
        } else if (startDate && selectedDate) {
            const [left, right] = sortDates(startDate, selectedDate);

            newRange = {
                startDate: left,
                endDate: right,
            };
        }

        this.handleRangeUpdate(newRange);
        this.checkFinishSelect(newRange);
    };

    private handleFocus = (): void => {
        const {onFocus} = this.props;
        const {isCalendarVisible} = this.state;

        if (!isCalendarVisible) {
            this.showPopup();
        }

        onFocus?.();
    };

    private handleMouseDown = (): void => {
        const {canToggle} = this.props;
        const {isCalendarVisible} = this.state;

        if (isCalendarVisible) {
            if (canToggle) {
                this.hidePopup();
            }
        } else {
            this.showPopup();
        }
    };

    private handleBlur = (): void => {
        const {onBlur} = this.props;

        this.hidePopup();

        onBlur?.();
    };

    private handleShowCalendar = (): void => {
        const {onShowCalendar} = this.props;

        onShowCalendar?.();
    };

    private handleHideCalendar = (): void => {
        const {onHideCalendar} = this.props;

        onHideCalendar?.();
    };

    private checkFinishSelect = ({
        startDate,
        endDate,
    }: INullableDateRange): void => {
        const {onFinishSelect, canAutoHideCalendar, canSelectRange} =
            this.props;

        if (!canAutoHideCalendar) {
            return;
        }

        onFinishSelect?.();

        if (startDate && (endDate || !canSelectRange)) {
            this.handleBlur();
        }
    };

    private getModalHistoryBackHandler() {
        if (this.backHandler) {
            return this.backHandler;
        }

        const {onModalHistoryBack} = this.props;

        if (!onModalHistoryBack) {
            return undefined;
        }

        this.backHandler = onModalHistoryBack.bind(this);

        return undefined;
    }

    private getSelectionMode(): EDatePickerSelectionMode {
        const {
            canSelectRange,
            startDate,
            endDate,
            resetRangeAfterFirstInteraction,
        } = this.props;
        const isNothingSelected = !startDate && !endDate;
        const isBothSelected = startDate && endDate;
        const isFirstInteraction =
            resetRangeAfterFirstInteraction &&
            !this.state.hasInteractionAfterShow;

        if (
            isNothingSelected ||
            isBothSelected ||
            !canSelectRange ||
            isFirstInteraction
        ) {
            return EDatePickerSelectionMode.START_DATE;
        }

        return EDatePickerSelectionMode.END_DATE;
    }

    /* Modal View */

    private renderModalCalendar = (): ReactNode => {
        const {isCalendarVisible} = this.state;

        if (!isCalendarVisible) {
            return null;
        }

        return (
            <DrawerContainer className={cx('drawerContainer')}>
                {this.renderDrawerHeader()}
                <Separator className={cx('separator')} />
                {this.renderCalendar(false)}
            </DrawerContainer>
        );
    };

    private renderDrawerHeader = (): React.ReactNode => {
        const {startDate, endDate, placeholder} = this.props;

        const showPlaceholder = !startDate && !endDate;

        let date = '';

        if (startDate) {
            date = formatDateRange(startDate, endDate || undefined, {
                forceHideFromDateYearInRange: true,
            });
        }

        const onResetClick = (): void => {
            if (startDate) {
                this.handleClickReset();
            }
        };

        return (
            <div className={cx('drawerHeader')}>
                <div className={cx('text')}>
                    {showPlaceholder ? placeholder : date}
                </div>
                <div>
                    <LinkButton
                        theme={showPlaceholder ? 'ghost' : 'normal'}
                        disabled={showPlaceholder}
                        onClick={onResetClick}
                        className={cx('resetButton')}
                    >
                        {i18nBlock.reset()}
                    </LinkButton>
                </div>
            </div>
        );
    };

    /* Dropdown View */

    private renderCalendarTrigger = (): ReactNode => {
        const {
            error,
            triggerClassName,
            triggerViewType,
            triggerFocusClassName,
            triggerValueClassName,
            triggerIconClassName,
            triggerPlaceholderClassName,
            triggerIcon,
            isShortTriggerFormatDate,
            canRenderResetIcon,
            deviceType: {isMobile},
            size,
            startDate,
            endDate,
            placeholder,
            triggerValue,
            tabIndex,
        } = this.props;
        const {hoveredDate, isCalendarVisible} = this.state;
        const selectionMode = this.getSelectionMode();

        const handleFocus = (): void => this.handleFocus();
        const handleMouseDown = (): void => this.handleMouseDown();

        return (
            <DatePickerTrigger
                className={cx(
                    triggerClassName,
                    'datePickerTrigger',
                    `datePickerTrigger_viewType_${triggerViewType}`,
                    {
                        datePickerTrigger_mobile: isMobile,
                    },
                )}
                ref={this.dateTriggerRef}
                focusClassName={triggerFocusClassName}
                valueClassName={triggerValueClassName}
                placeholderClassName={triggerPlaceholderClassName}
                iconClassName={triggerIconClassName}
                icon={triggerIcon}
                triggerViewType={triggerViewType}
                onMouseDown={handleMouseDown}
                onFocus={handleFocus}
                visiblePopup={isCalendarVisible}
                canRenderResetIcon={canRenderResetIcon}
                isShortFormatDate={isShortTriggerFormatDate}
                hasError={Boolean(error)}
                size={size}
                hoveredDate={hoveredDate}
                startDate={startDate}
                endDate={endDate}
                selectionMode={selectionMode}
                isMobile={isMobile}
                placeholder={placeholder}
                triggerValue={triggerValue}
                tabIndex={tabIndex}
                onClickReset={this.handleClickReset}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'trigger',
                })}
            />
        );
    };

    private onCalendarTouchStart = (e: TouchEvent): void => {
        if (!e.currentTarget) {
            return;
        }

        const {scrollTop} = e.currentTarget as HTMLElement;

        this.touchStartY = e.touches[0].clientY;
        this.scrollStartTop = scrollTop;
    };

    private onCalendarTouchMove = (e: TouchEvent): void => {
        const currentY = e.changedTouches[0].clientY;
        const startY = this.touchStartY as number;

        const swipeDown = startY < currentY;

        // Если календарь в самом начале и мы пытаемся проскролить еще вверх,
        // событие нужно отменить в браузерах с оверскролом
        if (e.cancelable && this.scrollStartTop === 0 && swipeDown) {
            e.preventDefault();

            // Считаем, что мы тянем шторку вниз только,
            // если пользователь за время события не свайпал вверх
            // Например, пользователь мог сначала прокрутить вверх, а потом чуть
            // прокрутить календарь вниз
            if (!this.hasSwipeUp) {
                this.drawerSwipeStarted = true;
            }
        }

        if (!swipeDown) {
            this.hasSwipeUp = true;
        }

        if (this.drawerSwipeStarted) {
            return;
        }

        // Если мы прокручиваем календарь, то событие не должно всплыть
        // в dragObserver шторки
        e.stopPropagation();
    };

    private onCalendarTouchEnd = (): void => {
        this.touchStartY = null;
        this.hasSwipeUp = false;
        this.drawerSwipeStarted = false;
    };

    private renderCalendarError = (): ReactNode => {
        const {isMobile} = this.props.deviceType;
        const {error, startDate, endDate} = this.props;
        const errorText = Array.isArray(error) && error[0];

        if (!isMobile || !errorText || !startDate || !endDate) {
            return null;
        }

        return <ErrorTooltip>{errorText}</ErrorTooltip>;
    };

    private renderDatePickerFooter = (): ReactNode => {
        const {footerBlockParams} = this.props;

        if (!footerBlockParams) {
            return null;
        }

        return (
            <DatePickerFooter
                {...footerBlockParams}
                onButtonClick={this.hideMobileDrawer}
            />
        );
    };

    private renderCalendarFooter = (): ReactNode => {
        return (
            <div className={cx('footer')}>
                {this.renderCalendarError()}
                {this.renderDatePickerFooter()}
            </div>
        );
    };

    private renderCalendar = (isRenderMonthsList: boolean): ReactNode => {
        const {
            nowDate,
            minDate,
            maxDate,
            startDate,
            endDate,
            cutPastWeeks,
            isModalView,
            canSelectRange,
            headerNode,
            withYearSelect,
        } = this.props;
        const selectionMode = this.getSelectionMode();
        const firstDate =
            selectionMode === EDatePickerSelectionMode.START_DATE && !endDate;

        return (
            <Calendar
                // @ts-ignore typeof не справляется
                ref={this.calendarRef}
                yearSelectRef={this.yearSelectRef}
                withYearSelect={withYearSelect}
                className={cx(
                    isModalView ? 'drawerCalendar' : 'dropdownCalendar',
                )}
                calendarWrapperClassName={cx(
                    isModalView && 'drawerCalendarWrapper',
                )}
                monthsGridClassName={cx(
                    isModalView ? 'drawerMonthsGrid' : 'dropdownMonthsGrid',
                )}
                weekdaysClassName={cx(
                    isModalView ? 'drawerWeekdays' : 'dropdownWeekdays',
                )}
                minDate={minDate}
                maxDate={maxDate}
                nowDate={nowDate}
                endDate={endDate}
                startDate={startDate}
                canSelectRange={firstDate ? false : canSelectRange}
                footerNode={this.renderCalendarFooter()}
                headerNode={headerNode}
                onDaySelect={this.handleDaySelect}
                onDayHover={this.handleDayHover}
                isRenderMonthsList={isRenderMonthsList}
                prices={this.props.prices}
                cutPastWeeks={cutPastWeeks}
                selectionMode={selectionMode}
                onMonthsGridTouchStart={this.onCalendarTouchStart}
                onMonthsGridTouchEnd={this.onCalendarTouchEnd}
                onMonthsGridTouchMove={this.onCalendarTouchMove}
                {...prepareQaAttributes('calendar')}
            />
        );
    };

    private renderPopup(): ReactNode {
        const {isCalendarVisible} = this.state;

        const {isModalView, size} = this.props;
        const onModalHistoryBack = this.getModalHistoryBackHandler();

        if (isModalView) {
            return (
                <WithHistoryBack
                    isVisible={isCalendarVisible}
                    close={this.hidePopup}
                    onHistoryBack={onModalHistoryBack}
                >
                    <ModalOrBottomSheet
                        isVisible={isCalendarVisible}
                        modalProps={{
                            isMobile: true,
                        }}
                        bottomSheetProps={{
                            contentClassName: cx('bottomSheetContent'),
                            padding: 0,
                            keepMounted: false,
                        }}
                        onClose={this.hideMobileDrawer}
                    >
                        <div ref={this.popupRef}>
                            {this.renderModalCalendar()}
                        </div>
                    </ModalOrBottomSheet>
                </WithHistoryBack>
            );
        }

        if (!isCalendarVisible) {
            return null;
        }

        const dateTriggerRef = this.dateTriggerRef;
        const anchor = dateTriggerRef?.current?.buttonRef?.current;

        if (!anchor) {
            return null;
        }

        return (
            <ScopeContext.Consumer>
                {(scope): ReactNode => (
                    <Popup
                        className={cx('popupContainer')}
                        visible
                        nonvisual={false}
                        anchor={{current: anchor}}
                        directions={POPUP_DIRECTIONS}
                        secondaryOffset={
                            size === 'xl'
                                ? MONTH_LIST_WIDTH + TRIGGER_OFFSET
                                : TRIGGER_OFFSET
                        }
                        mainOffset={size === 'xl' ? undefined : 8}
                        zIndex={105}
                        scope={scope ? {current: scope} : undefined}
                        plain
                    >
                        <div ref={this.popupRef}>
                            {this.renderCalendar(true)}
                        </div>
                    </Popup>
                )}
            </ScopeContext.Consumer>
        );
    }

    render(): ReactNode {
        const {className, size} = this.props;

        return (
            <div
                className={cx(
                    'datePicker',
                    'datePickerNew',
                    `datePicker_${size}`,
                    className,
                )}
                {...prepareQaAttributes(this.props)}
            >
                {this.renderCalendarTrigger()}
                {this.renderPopup()}
            </div>
        );
    }
}

export default DatePickerNew;
