import React, {
    ComponentType,
    MouseEventHandler,
    PureComponent,
    ReactNode,
} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EDatePickerSelectionMode} from 'components/DatePicker/types';
import {IIconProps} from 'icons/types/icon';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getDateTriggerTextParams from './utilities/getDateTriggerTextParams';
import getFormattedTextValue from 'components/DatePicker/components/DatePickerTrigger/utilities/getFormattedTextValue';
import {CHAR_EM_DASH, CHAR_NBSP} from 'utilities/strings/charCodes';

import {TDatePickerSize} from 'components/DatePicker/DatePicker';
import SearchControl, {
    ESearchControlPlaceholderPosition,
} from 'components/SearchControl/SearchControl';
import CalendarIcon from 'icons/16/Calendar';
import DatePickerIcon from 'icons/16/DatePicker';
import CloseCircleIcon from 'icons/16/CloseCircle';

import cx from './DatePickerTrigger.scss';

export type TDatePickerTriggerValueFn = (params: {
    startDate?: Date | string | null;
    endDate?: Date | string | null;
    hoveredDate?: Date | string | null;
}) => string | undefined;

export interface IDatePickerTriggerProps
    extends IWithClassName,
        IWithQaAttributes {
    focusClassName: string;
    valueClassName?: string;
    placeholderClassName?: string;
    iconClassName?: string;
    icon?: ComponentType<IIconProps>;
    placeholder: string;
    visiblePopup: boolean;
    startDate: Nullable<Date>;
    endDate: Nullable<Date>;
    hoveredDate: Nullable<Date>;
    selectionMode: EDatePickerSelectionMode;
    triggerViewType: 'union' | 'tile';
    size: TDatePickerSize;
    isShortFormatDate: boolean;
    isMobile: boolean;
    canRenderResetIcon: boolean;
    tabIndex?: number;
    hasError: boolean;
    triggerValue?: string | TDatePickerTriggerValueFn;
    onClick?: () => void;
    onMouseDown?: () => void;
    onFocus?: () => void;
    onBlur?: () => void;
    onKeyDown?: () => void;
    onClickReset?: () => void;
}

class DatePickerTriggerNew extends PureComponent<IDatePickerTriggerProps> {
    buttonRef = React.createRef<HTMLButtonElement>();
    labelRef = React.createRef<HTMLLabelElement>();

    static defaultProps = {
        className: '',
        focusClassName: '',
        placeholder: '',
        triggerViewType: 'union',
        size: 'xl',
        visiblePopup: false,
        isShortFormatDate: true,
        canRenderResetIcon: true,
        startDate: null,
        endDate: null,
        hoveredDate: null,
        isMobile: false,
        hasError: false,
    };

    focus(): void {
        const {current} = this.buttonRef;

        if (current) {
            current.focus();
        }
    }

    blur(): void {
        const {current} = this.buttonRef;

        if (current) {
            current.blur();
        }
    }

    private handleClickReset: MouseEventHandler = e => {
        const {onClickReset} = this.props;

        onClickReset?.();

        e.preventDefault();
        e.stopPropagation();

        this.focus();
    };

    private renderValue(): ReactNode {
        const {placeholder, size} = this.props;

        const valueNode = this.renderValueNode();

        if (size === 'xl') {
            return (
                <>
                    {this.renderButton()}
                    <SearchControl
                        placeholderClassName={cx('placeholder')}
                        placeholderPosition={
                            ESearchControlPlaceholderPosition.BOTTOM
                        }
                        placeholder={placeholder}
                        controlNode={valueNode}
                        isEmpty={!valueNode}
                    />
                </>
            );
        }

        return this.renderButton(valueNode || this.renderPlaceholderNode());
    }

    private renderValueNodeText(): ReactNode {
        const {startDate, endDate, hoveredDate, selectionMode} = this.props;

        const {left, right, highlightRight, highlightLeft} =
            getDateTriggerTextParams({
                startDate,
                endDate,
                hoveredDate,
                selectionMode,
            });

        if (!left) {
            return null;
        }

        const delimiter = `${CHAR_NBSP}${CHAR_EM_DASH}${CHAR_NBSP}`;
        const [leftFormatted, rightFormatted] = getFormattedTextValue(
            left,
            right,
        );

        return (
            <>
                {leftFormatted && (
                    <span className={cx({valueHovered: highlightLeft})}>
                        {leftFormatted}
                    </span>
                )}
                {rightFormatted && (
                    <span className={cx({valueHovered: highlightRight})}>
                        {delimiter}
                        {rightFormatted}
                    </span>
                )}
            </>
        );
    }

    private getPreformattedValue(): string | undefined {
        const {startDate, endDate, triggerValue, hoveredDate} = this.props;

        if (typeof triggerValue === 'function') {
            return triggerValue({startDate, hoveredDate, endDate});
        }

        return triggerValue;
    }

    private renderValueNode(): ReactNode {
        const {startDate, endDate, hoveredDate, valueClassName} = this.props;

        const preformatted = this.getPreformattedValue();

        const isEmptyValue =
            !startDate && !endDate && !preformatted && !hoveredDate;

        return (
            !isEmptyValue && (
                <div
                    className={cx('value', valueClassName)}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'trigger-value',
                    })}
                >
                    {preformatted || this.renderValueNodeText()}
                </div>
            )
        );
    }

    private renderPlaceholderNode(): ReactNode {
        const {placeholder, placeholderClassName} = this.props;

        return (
            <span className={cx('placeholder', placeholderClassName)}>
                {placeholder}
            </span>
        );
    }

    private renderResetIcon(): ReactNode {
        const {startDate, endDate, canRenderResetIcon} = this.props;

        const triggerValue = this.getPreformattedValue();

        if (canRenderResetIcon && (startDate || endDate || triggerValue)) {
            return (
                <div
                    className={cx('resetIcon')}
                    onClick={this.handleClickReset}
                >
                    <CloseCircleIcon
                        className={cx('crossIcon')}
                        width={18}
                        height={18}
                    />
                </div>
            );
        }

        return null;
    }

    private renderDatePickerIcon(): ReactNode {
        const {
            canRenderResetIcon,
            visiblePopup,
            startDate,
            endDate,
            size,
            iconClassName,
            icon: Icon,
        } = this.props;

        const triggerValue = this.getPreformattedValue();

        if (canRenderResetIcon && (startDate || endDate || triggerValue)) {
            return null;
        }

        if (Icon) {
            return (
                <Icon className={cx('datePickerIcon__icon', iconClassName)} />
            );
        }

        if (size === 'xl') {
            return (
                <div className={cx('datePickerIcon')}>
                    <DatePickerIcon
                        className={cx('datePickerIcon__icon', iconClassName, {
                            active: visiblePopup,
                        })}
                    />
                </div>
            );
        }

        return (
            <CalendarIcon
                className={cx('datePickerIcon__icon', iconClassName)}
            />
        );
    }

    private renderButton(buttonContent?: ReactNode): ReactNode {
        const {onFocus, onBlur, onKeyDown, tabIndex} = this.props;

        return (
            <button
                className={cx('datePickerTriggerButton')}
                type="button"
                tabIndex={tabIndex}
                onFocus={onFocus}
                onBlur={onBlur}
                onKeyDown={onKeyDown}
                ref={this.buttonRef}
            >
                {buttonContent}
            </button>
        );
    }

    render(): ReactNode {
        const {
            className,
            visiblePopup,
            isMobile,
            triggerViewType,
            focusClassName,
            hasError,
            size,
            onClick,
            onMouseDown,
        } = this.props;

        return (
            <label
                className={cx(
                    'datePickerTrigger',
                    `datePickerTrigger_${size}`,
                    {
                        datePickerTrigger_focus: visiblePopup,
                        datePickerTrigger_error: hasError,
                        datePickerTrigger_mobile: isMobile,
                        [`datePickerTrigger_viewType_${triggerViewType}`]:
                            triggerViewType,
                    },
                    className,
                    visiblePopup && focusClassName,
                )}
                ref={this.labelRef}
                onClick={onClick}
                onMouseDown={onMouseDown}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'trigger',
                })}
            >
                {this.renderValue()}
                {this.renderDatePickerIcon()}
                {this.renderResetIcon()}
            </label>
        );
    }
}

export default DatePickerTriggerNew;
