import {Component, CSSProperties, ReactNode} from 'react';
import {noop} from 'lodash';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import AdsScript from 'components/AdsScript/AdsScript';

import cx from './Direct.scss';

export interface IDirectProps extends IWithClassName, IWithQaAttributes {
    blockId: string;
    isMobile?: boolean;
    onLoad?: (data: IDataForCallbacks, ...args: any[]) => void; // будет вызвана, когда рекламный блок отрендерится
    onNotRender?: (data: IDataForCallbacks) => void; // будет вызвана в том случае, если баннер был запрошен,
    // но система директа ничего не вернула, т.е. не подобрала нужный баннер для страницы
    blockIdStub?: string; // другая реклама на случай, если blockId не вернулась, вызывается после onNotRender
    onClose?: (data: IDataForCallbacks, event: Event) => void; // будет вызван, если пользователь закрыл баннер
    refCallback?: (element: HTMLElement) => void;
    updateTimeoutInSecond: number;
    style?: CSSProperties;
}

export interface IDataForCallbacks {
    elementId: string;
    blockId: string;
}

interface IDirectState {
    blockId: string;
    pageNumber?: number;
    requestingBlockId?: string;
    mounted: boolean;
}

interface IRootElementAttributes {
    id?: string;
}

let pageCounter = 0;

const EVENT_CLOSE = 'ya-internal-close';
const MILLISECONDS_IN_SECOND = 1000;

export class Direct extends Component<IDirectProps, IDirectState> {
    private static makeContainerId = (state: IDirectState): string => {
        return `banner-${state.blockId}-${state.pageNumber}`;
    };

    private onCloseListeners: EventListenerOrEventListenerObject[] = [];
    private domElement?: HTMLDivElement;
    private lastUpdateTime: number = Number.NEGATIVE_INFINITY;

    static readonly defaultProps: Partial<IDirectProps> = {
        isMobile: false,
        updateTimeoutInSecond: 50,
    };

    constructor(props: IDirectProps) {
        super(props);
        this.state = {
            blockId: this.props.blockId,
            mounted: false,
        };
    }

    componentDidMount(): void {
        this.setState({
            mounted: true,
            pageNumber: pageCounter++,
        });

        const {onClose = noop} = this.props;
        const dataForCallbacks = this.getDataForCallbacks();

        this.addOnCloseListener(e => onClose(dataForCallbacks, e));
    }

    componentDidUpdate(prevProps: IDirectProps, prevState: IDirectState): void {
        if (
            Direct.makeContainerId(this.state) !==
            Direct.makeContainerId(prevState)
        ) {
            this.renderDirect();
        }
    }

    componentWillUnmount(): void {
        this.removeOnCloseListeners();
    }

    update = (force?: boolean): void => {
        const {updateTimeoutInSecond} = this.props;
        const now = Date.now();

        if (
            !force &&
            (now - this.lastUpdateTime) / MILLISECONDS_IN_SECOND <=
                updateTimeoutInSecond
        ) {
            return;
        }

        this.setState({pageNumber: pageCounter++});
    };

    private addOnCloseListener = (
        listener: EventListenerOrEventListenerObject,
    ): EventListenerOrEventListenerObject[] | void => {
        if (!this.domElement) {
            return;
        }

        this.domElement.addEventListener(EVENT_CLOSE, listener);
        this.onCloseListeners.push(listener);

        return this.onCloseListeners;
    };

    private removeOnCloseListeners = ():
        | EventListenerOrEventListenerObject[]
        | void => {
        if (!this.domElement) {
            return;
        }

        this.onCloseListeners.forEach(listener => {
            // @ts-ignore
            this.domElement.removeEventListener(EVENT_CLOSE, listener);
        });
        this.onCloseListeners = [];

        return this.onCloseListeners;
    };

    private getContainerId = (): string => {
        return Direct.makeContainerId(this.state);
    };

    private getDataForCallbacks = (): IDataForCallbacks => {
        return {
            elementId: this.getContainerId(),
            blockId: this.state.blockId,
        };
    };

    private renderDirect = (): void => {
        const {onLoad = noop, onNotRender = noop} = this.props;
        const {pageNumber, blockId, requestingBlockId} = this.state;
        const containerId = this.getContainerId();
        const dataForCallbacks = this.getDataForCallbacks();

        if (requestingBlockId !== blockId) {
            this.setState({
                requestingBlockId: blockId,
            });

            (window.yaContextCb ||= []).push(() => {
                window.Ya.Context.AdvManager.render(
                    {
                        blockId,
                        pageNumber,
                        renderTo: containerId,
                        async: true,
                        onRender: (...args) => {
                            this.setState({
                                requestingBlockId: undefined,
                            });

                            this.lastUpdateTime = Date.now();

                            onLoad(dataForCallbacks, ...args);
                        },
                    },
                    () => {
                        onNotRender(dataForCallbacks);
                        this.tryRequestStubDirect();
                    },
                );
            });
        }
    };

    private tryRequestStubDirect(): void {
        if (
            this.props.blockIdStub &&
            this.state.blockId !== this.props.blockIdStub
        ) {
            this.setState({blockId: this.props.blockIdStub});
        }
    }

    private refCallback = (element: HTMLDivElement): void => {
        const {refCallback: refCallbackFromProps} = this.props;

        this.domElement = element;

        if (refCallbackFromProps) {
            refCallbackFromProps(element);
        }
    };

    render(): ReactNode {
        const {className, isMobile, style} = this.props;
        const {mounted} = this.state;
        const params: IRootElementAttributes = {};

        if (mounted) {
            params.id = this.getContainerId();
        }

        return (
            <>
                <AdsScript />

                <div
                    {...prepareQaAttributes(this.props)}
                    className={cx('root', className, {mobile: isMobile})}
                    ref={this.refCallback}
                    style={style}
                    {...params}
                />
            </>
        );
    }
}
