import React, {
    Component,
    ComponentType,
    createElement,
    ReactNode,
    MouseEvent as ReactMouseEvent,
    KeyboardEvent as ReactKeyboardEvent,
    MutableRefObject,
} from 'react';
import _noop from 'lodash/noop';
import _get from 'lodash/get';

import {TAB_KEY_CODE} from 'constants/eventKeyCodes';

import {IWithClassName} from 'types/withClassName';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Popup, {IPopupProps} from 'components/Popup/Popup';
import {MODAL_LAYER_MULTIPLIER} from 'components/Modal/Modal';

import ScopeContext from 'contexts/ScopeContext';
import LayersContext from 'contexts/LayersContext';

export interface IMetaParams {
    hidePopup(): void;
    showPopup(): void;
    visible: boolean;
}

interface ISwitcherInput extends IWithClassName {
    onFocus(): void;
    onBlur(): void;
    onKeyDown(evt: ReactKeyboardEvent): void;
    onMouseDown(evt: ReactMouseEvent): void;
    onClick(evt: ReactMouseEvent): void;
    ref(switcherNode: HTMLElement | null): void;
}

export interface IDropdownSwitcherParams<T = {}> {
    componentProps: T;
    meta: IMetaParams;
    input: ISwitcherInput;
}

export interface IDropdownPopupParams<T = {}> {
    componentProps: T;
    meta: IMetaParams;
}

export interface IDropdownProps<
    SwitcherComponentProps = {},
    PopupComponentProps = {},
> {
    switcherClassName?: string;
    /** Switcher is rendered via React.createElement. Takes precedence over renderSwitcher. */
    switcherComponent?: ComponentType<
        IDropdownSwitcherParams<SwitcherComponentProps>
    >;
    renderSwitcher?(
        params: IDropdownSwitcherParams<SwitcherComponentProps>,
    ): ReactNode;
    switcherComponentProps: SwitcherComponentProps;

    popupClassName?: string;
    /** Popup content is rendered via React.createElement. Takes precedence over renderPopup. */
    popupComponent?: ComponentType<IDropdownPopupParams<PopupComponentProps>>;
    renderPopup?(params: IDropdownPopupParams<PopupComponentProps>): ReactNode;
    popupProps?: Partial<IPopupProps>;
    popupContainerClassName?: string;
    popupComponentProps: PopupComponentProps;
    /** Fixes popup position after opening */
    popupMotionless?: boolean;

    visible: boolean;
    onShowPopup(): void;
    onHidePopup(): void;
    onFocus(): void;
    onBlur(): void;

    isModalView: boolean;
    onModalHistoryBack?(): void;
    initialBodyOverflow?: string;
    canToggle?: boolean;
    popupRef?: MutableRefObject<HTMLElement | null>;
}

interface IDropdownState {
    visible: boolean;
    shouldHandleShow: boolean;
}

// TODO 2020-03-06 ndru: Есть проблема с Dropdown: вызов setSwitcherRef происходит позже,
//  чем рендер PopupWithHistoryBack (в него этот реф прокидывается как anchor для Popup).
//  Из-за отсутствия anchor Popup не маунтит ноду-обёртку для контента, и в итоге, например, при фокусировании
//  элемента в контенте попапа происходит скролл в самый низ страницы.
//  @see https://st.yandex-team.ru/TRAVELFRONT-2636

class Dropdown<
    SwitcherComponentProps extends {} = {},
    PopupComponentProps extends {} = {},
> extends Component<
    IDropdownProps<SwitcherComponentProps, PopupComponentProps>,
    IDropdownState
> {
    private isFocused = false;
    private popupNode: HTMLElement | null = null;
    private switcherNode: HTMLElement | null = null;
    private initialBodyOverflowValue: string | null = null;

    static readonly defaultProps = {
        visible: false,
        onShowPopup: _noop,
        onHidePopup: _noop,
        onModalHistoryBack: _noop,
        onFocus: _noop,
        onBlur: _noop,
        switcherClassName: '',
        popupClassName: '',
        isModalView: false,
        popupComponentProps: {},
        popupContainerClassName: '',
        switcherComponentProps: {},
        popupProps: {},
    };

    constructor(
        props: IDropdownProps<SwitcherComponentProps, PopupComponentProps>,
        context?: unknown,
    ) {
        super(props, context);

        this.state = {
            visible: false,
            shouldHandleShow: false,
        };
    }

    componentDidMount(): void {
        this.setInitialBodyOverflowValue();
        this.checkShowPopup();
    }

    componentDidUpdate(
        prevProps: IDropdownProps<SwitcherComponentProps, PopupComponentProps>,
    ): void {
        if (this.props.visible !== prevProps.visible) {
            (this.props.visible ? this.showPopup : this.hidePopup)();
        }
    }

    componentWillUnmount(): void {
        this.unBindEvents();
    }

    isVisible(): boolean {
        return this.state.visible;
    }

    /* Refs manipulate */

    private getSwitcherRef = (): HTMLElement | null => {
        return this.switcherNode;
    };

    private setSwitcherRef = (switcherNode: HTMLElement | null): void => {
        this.switcherNode = switcherNode;
    };

    private getPopupRef = (): HTMLElement | null => {
        return this.popupNode;
    };

    private setPopupRef = (popupNode: HTMLElement | null): void => {
        this.popupNode = popupNode;

        if (this.props.popupRef) {
            this.props.popupRef.current = popupNode;
        }
    };

    /* Bindings */

    private initBindings = (): void => {
        document.addEventListener('mousedown', this.handleOutsideInteraction);
        document.addEventListener('focus', this.handleOutsideInteraction, true);
    };

    private unBindEvents = (): void => {
        document.removeEventListener(
            'mousedown',
            this.handleOutsideInteraction,
        );
        document.removeEventListener(
            'focus',
            this.handleOutsideInteraction,
            true,
        );
    };

    /* Events handlers */

    private handleOutsideInteraction = ({target}: Event): void => {
        const {isModalView} = this.props;
        const {visible} = this.state;

        if (visible && isModalView === false) {
            const targetNode = target instanceof Node ? target : null;
            const switcherNode = this.getSwitcherRef();
            const popupNode = this.getPopupRef();

            const isSwitcherMouseEvent =
                switcherNode && switcherNode.contains(targetNode);
            const isPopupMouseEvent =
                popupNode && popupNode.contains(targetNode);

            if (!(isSwitcherMouseEvent || isPopupMouseEvent)) {
                this.handleBlur();
            }
        }
    };

    private handleFocus = (): void => {
        // не даем фокусу при тоггле переоткрыть дропдаун после его закрытия
        if (this.props.canToggle && this.isFocused) return;

        this.isFocused = true;
        this.props.onFocus();

        if (!this.state.visible) {
            this.showPopup();
        }
    };

    private handleBlur = (): void => {
        this.isFocused = false;
        this.props.onBlur();
        this.hidePopup();
    };

    private handleMouseDown = (): void => {
        this.tryToggle();
    };

    private handleKeyDown: React.KeyboardEventHandler<Element> = ({
        keyCode,
    }) => {
        switch (keyCode) {
            case TAB_KEY_CODE: {
                this.hidePopup();

                break;
            }

            default: {
            }
        }
    };

    private focus = (): void => {
        if (!this.isFocused) {
            const switcherNode = this.getSwitcherRef();

            if (switcherNode) {
                switcherNode.focus();
            }
        }
    };

    private tryToggle = (): void => {
        if (this.state.visible) {
            if (this.props.canToggle) {
                this.hidePopup();
            }
        } else {
            this.showPopup();
        }
    };

    showPopup = (): void => {
        if (!this.state.visible) {
            this.setState(
                ({visible}) => {
                    if (visible) {
                        return {
                            visible: false,
                            shouldHandleShow: false,
                        };
                    }

                    return {
                        visible: true,
                        shouldHandleShow: true,
                    };
                },
                () => {
                    if (this.state.shouldHandleShow) {
                        this.props.onShowPopup();
                        this.preventBodyScroll();
                        this.initBindings();
                        this.focus();
                    }
                },
            );
        }
    };

    hidePopup = (): void => {
        const {visible} = this.state;
        const {onHidePopup} = this.props;

        if (visible) {
            this.setState({visible: false});
            onHidePopup();
            this.revertBodyScroll();
            this.unBindEvents();
        }
    };

    private checkShowPopup = (): void => {
        const {visible} = this.props;

        if (visible) {
            this.showPopup();
        }
    };

    private setInitialBodyOverflowValue = (): void => {
        const {isModalView, initialBodyOverflow} = this.props;

        if (isModalView) {
            this.initialBodyOverflowValue =
                initialBodyOverflow ??
                (_get(document, 'body.style.overflow') || '');
        }
    };

    private preventBodyScroll = (): void => {
        const {isModalView} = this.props;

        if (isModalView) {
            document.body.style.overflow = 'hidden';
        }
    };

    private revertBodyScroll = (): void => {
        document.body.style.overflow = this.initialBodyOverflowValue!;
    };

    private preparePropsForRenderSwitcher(): IDropdownSwitcherParams<SwitcherComponentProps> {
        const {visible} = this.state;
        const {switcherClassName, switcherComponentProps} = this.props;

        return {
            meta: {
                visible,
                hidePopup: this.hidePopup,
                showPopup: this.showPopup,
            },
            input: {
                onFocus: this.handleFocus,
                onBlur: this.handleBlur,
                onKeyDown: this.handleKeyDown,
                onMouseDown: this.handleMouseDown,
                //Только для тачей см https://st.yandex-team.ru/TRAVELFRONT-4622
                onClick: this.handleMouseDown,
                ref: this.setSwitcherRef,
                className: switcherClassName,
            },
            componentProps: switcherComponentProps,
        };
    }

    private renderSwitcher(): ReactNode {
        const {switcherComponent, renderSwitcher} = this.props;
        const switcherProps = this.preparePropsForRenderSwitcher();

        if (switcherComponent) {
            return createElement(switcherComponent, switcherProps);
        }

        if (renderSwitcher) {
            return renderSwitcher(switcherProps);
        }

        return null;
    }

    private renderPopup(): ReactNode {
        const {visible} = this.state;
        const {
            popupProps,
            popupClassName,
            popupContainerClassName,
            popupMotionless,
            isModalView,
            onModalHistoryBack,
        } = this.props;

        if (isModalView) {
            return (
                <ModalWithHistoryBack
                    isMobile
                    unmountOnHide
                    fullScreen
                    hasCloseButton={false}
                    className={popupClassName}
                    isVisible={visible}
                    onClose={this.hidePopup}
                    onHistoryBack={onModalHistoryBack}
                >
                    <div
                        className={popupContainerClassName}
                        ref={this.setPopupRef}
                    >
                        {this.renderPopupContent()}
                    </div>
                </ModalWithHistoryBack>
            );
        }

        return (
            <ScopeContext.Consumer>
                {(scope): ReactNode => (
                    <LayersContext.Consumer>
                        {(layer): React.ReactNode => (
                            <Popup
                                className={popupClassName}
                                visible={visible}
                                motionless={popupMotionless}
                                anchor={{current: this.getSwitcherRef()}}
                                zIndex={layer * MODAL_LAYER_MULTIPLIER}
                                scope={scope ? {current: scope} : undefined}
                                {...popupProps}
                            >
                                <div
                                    className={popupContainerClassName}
                                    ref={this.setPopupRef}
                                >
                                    {this.renderPopupContent()}
                                </div>
                            </Popup>
                        )}
                    </LayersContext.Consumer>
                )}
            </ScopeContext.Consumer>
        );
    }

    private renderPopupContent(): ReactNode {
        const {visible} = this.state;
        const {popupComponent, popupComponentProps, renderPopup} = this.props;
        const popupProps = {
            meta: {
                hidePopup: this.hidePopup,
                showPopup: this.showPopup,
                visible,
            },
            componentProps: popupComponentProps,
        };

        if (popupComponent) {
            return createElement(popupComponent, popupProps);
        }

        if (renderPopup) {
            return renderPopup(popupProps);
        }

        return null;
    }

    render(): ReactNode {
        return (
            <>
                {this.renderSwitcher()}
                {this.renderPopup()}
            </>
        );
    }
}

export default Dropdown;
