import React, {ReactNode, useCallback, useRef} from 'react';
import Foco from 'react-foco';

import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {IWithClassName} from 'types/withClassName';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import CheckButton from 'components/CheckButton/CheckButton';
import Popup from 'components/Popup/Popup';
import ArrowBottomIcon from 'icons/16/ArrowBottom';

import cx from './DropdownFilter.scss';

interface IDropdownFilterProps extends IWithQaAttributes, IWithClassName {
    popupClassName?: string;
    label: React.ReactNode;
    checked?: boolean;
    children: React.ReactNode;
    /** Клик по кнопке-открывашке */
    onClickButton?: () => void;
    withArrowsIcon?: boolean;
}

const DIRECTIONS = [EPopupDirection.BOTTOM_LEFT, EPopupDirection.BOTTOM_RIGHT];

const DropdownFilter: React.FC<IDropdownFilterProps> = props => {
    const {
        label,
        children,
        checked,
        popupClassName,
        onClickButton,
        withArrowsIcon = true,
        className,
    } = props;

    const {
        value: visible,
        setFalse: hidePopup,
        setTrue: showPopup,
    } = useBoolean(false);

    const handlerClickCheckButton = useCallback(() => {
        if (visible) {
            hidePopup();
        } else {
            showPopup();
        }

        if (onClickButton) {
            onClickButton();
        }
    }, [hidePopup, onClickButton, showPopup, visible]);

    const anchorRef = useRef<HTMLElement>(null);
    const containerRef = useRef<HTMLElement>(null);

    return (
        <Foco
            onClickOutside={hidePopup}
            onFocusOutside={hidePopup}
            className={className}
        >
            {(focoProps): ReactNode => (
                <span
                    className={cx('trigger-container')}
                    ref={containerRef}
                    {...focoProps}
                >
                    <CheckButton
                        {...prepareQaAttributes(props)}
                        checked={Boolean(checked)}
                        size="m-inset"
                        innerRef={anchorRef}
                        onClick={handlerClickCheckButton}
                    >
                        {label}
                        {withArrowsIcon && (
                            <ArrowBottomIcon
                                className={cx('icon', {icon_open: visible})}
                            />
                        )}
                    </CheckButton>

                    <Popup
                        className={cx('popup', popupClassName)}
                        visible={visible}
                        scope={containerRef}
                        anchor={anchorRef}
                        directions={DIRECTIONS}
                        zIndex={10}
                    >
                        {children}
                    </Popup>
                </span>
            )}
        </Foco>
    );
};

export default React.memo(DropdownFilter);
