import React, {useEffect} from 'react';

export interface IDynamicScriptProps {
    src: string;
    onLoad?(): void;
    onError?(): void;
    target?: React.RefObject<HTMLElement>;
    attrs?: Record<string, string | undefined>;
}

const addedScripts = new Set<string>();

const DynamicScript: React.FC<IDynamicScriptProps> = props => {
    const {src, attrs, onLoad, onError, target} = props;

    useEffect(() => {
        if (addedScripts.has(src)) {
            return;
        }

        const script = document.createElement('script');

        script.src = src;
        script.type = 'text/javascript';

        if (attrs) {
            Object.entries(attrs).forEach(([attrKey, attrVal]) => {
                if (attrVal !== undefined) {
                    script.setAttribute(attrKey, attrVal);
                }
            });
        }

        if (onLoad) {
            script.onload = onLoad;
        }

        if (onError) {
            script.onerror = onError;
        }

        if (target && target.current) {
            target.current.appendChild(script);
        } else {
            document.head.appendChild(script);
        }

        addedScripts.add(src);
    }, [attrs, onError, onLoad, src, target]);

    return null;
};

export default DynamicScript;
