import React, {useCallback, useEffect, useRef, useState} from 'react';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Text from 'components/Text/Text';
import Modal from 'components/Modal/Modal';
import Heading from 'components/Heading/Heading';
import Link, {TLinkProps} from 'components/Link/Link';
import Separator from 'components/Separator/Separator';
import Button, {IButtonProps} from 'components/Button/Button';
import ButtonLink, {IButtonLinkProps} from 'components/ButtonLink/ButtonLink';

import cx from './ErrorModal.scss';

interface IBaseErrorAction {
    title: string;
    handler?: React.MouseEventHandler;
}

export interface IErrorActionButton extends IBaseErrorAction {
    type: 'button';
    props?: Omit<IButtonProps, 'theme' | 'size' | 'onClick'>;
}

interface IErrorActionButtonLink extends IBaseErrorAction {
    type: 'button-link';
    props?: Omit<IButtonLinkProps, 'theme' | 'size' | 'onClick'>;
}

export type TErrorActionType = IErrorActionButton | IErrorActionButtonLink;

export interface IErrorModalProps extends IWithQaAttributes {
    /**
     * Заголовок (жирным текстом)
     */
    title: string;

    /**
     * Основной текст
     */
    text?: string | React.ReactElement;

    /**
     * Признак видимости окна
     */
    isVisible: boolean;

    /**
     * Данные ссылки
     */
    link?: TLinkProps;

    /**
     * Компонент с картинкой
     */
    image?: React.ReactNode;

    /**
     * Кнопка основного действия
     */
    primaryAction: TErrorActionType;

    /**
     * Кнопка второстепенного действия
     */
    secondaryAction?: TErrorActionType;

    onMount?(): void;

    /**
     * Можно ли закрыть попап
     */
    isClosable?: boolean;

    /**
     * коллбэк закрытия
     */
    onClose?(): void;
}

const ErrorModal: React.FC<IErrorModalProps> = props => {
    const {
        title,
        text,
        image,
        link,
        children,
        primaryAction,
        secondaryAction,
        isVisible,
        onMount,
        isClosable = false,
        onClose,
    } = props;
    const deviceType = useDeviceType();
    const timeoutRef = useRef<number>();
    const [shaked, letsShake] = useState(false);

    const handleNonActionClosing = useCallback(() => {
        if (!shaked) {
            letsShake(true);

            timeoutRef.current = window.setTimeout(() => {
                letsShake(false);
            }, 1000);
        }
    }, [shaked, letsShake]);

    useEffect(
        () => (): void => {
            clearTimeout(timeoutRef.current);
        },
        [],
    );

    useEffect(() => {
        onMount?.();
    }, [onMount]);

    const {isMobile} = deviceType;

    const PrimaryActionComponent =
        primaryAction.type === 'button' ? Button : ButtonLink;
    const SecondaryActionComponent =
        secondaryAction && secondaryAction.type === 'button'
            ? Button
            : ButtonLink;

    return (
        <Modal
            containerClassName={cx({shaked})}
            isMobile={isMobile}
            isVisible={isVisible}
            hasCloseButton={isClosable}
            onClose={isClosable ? onClose : handleNonActionClosing}
        >
            <div
                className={cx('root', deviceMods('root', deviceType))}
                {...prepareQaAttributes(props)}
            >
                {image && <div className={cx('image')}>{image}</div>}
                <Heading
                    className={cx('title')}
                    level="2"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'title',
                    })}
                >
                    {title}
                </Heading>
                {text && (
                    <Text
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'text',
                        })}
                    >
                        {text}
                    </Text>
                )}
                {children && (
                    <div
                        className={cx('content')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'content',
                        })}
                    >
                        {children}
                    </div>
                )}

                <div className={cx('buttonsContainer')}>
                    <div className={cx('buttons')}>
                        {secondaryAction && (
                            <SecondaryActionComponent
                                {...secondaryAction.props}
                                theme="secondary"
                                size={isMobile ? 'l' : 'm-inset'}
                                className={cx(
                                    'button',
                                    secondaryAction.props?.className,
                                )}
                                onClick={secondaryAction.handler}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'secondaryAction',
                                })}
                            >
                                {secondaryAction.title}
                            </SecondaryActionComponent>
                        )}

                        <Modal.AutoFocus className={cx('button')}>
                            <PrimaryActionComponent
                                {...primaryAction.props}
                                theme="primary"
                                width={isMobile ? 'max' : 'auto'}
                                size={isMobile ? 'l' : 'm-inset'}
                                className={cx(
                                    'button',
                                    primaryAction.props?.className,
                                )}
                                onClick={primaryAction.handler}
                                {...prepareQaAttributes({
                                    parent: props,
                                    current: 'primaryAction',
                                })}
                            >
                                {primaryAction.title}
                            </PrimaryActionComponent>
                        </Modal.AutoFocus>
                    </div>
                </div>

                {isMobile && link && (
                    <Separator className={cx('separator')} margin="5" />
                )}
                {link && (
                    <div className={cx('link')}>
                        <Link {...link} />
                    </div>
                )}
            </div>
        </Modal>
    );
};

export default ErrorModal;
