import {ReactNode, useCallback, useState} from 'react';
import {Meta} from '@storybook/react/types-6-0';

import {TStorybookFunctionComponent} from 'storybook/types';

import {CurrencyType} from 'utilities/currency/CurrencyType';

import Price from 'components/Price/Price';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import FilterCheckList from './FilterCheckList';

export default {
    title: 'Example/FilterCheckList',
    component: FilterCheckList,
} as Meta;

export const Simple: TStorybookFunctionComponent = () => {
    const [values, setValues] = useState<string[]>([]);

    const handleChange = useCallback(
        (option: string, checked: boolean) => {
            const newValues = checked
                ? [...values, option]
                : values.filter(value => value !== option);

            setValues(newValues);
        },
        [values],
    );

    return (
        <DeviceTypeContext.Consumer>
            {(deviceType): ReactNode => (
                <FilterCheckList
                    deviceType={deviceType}
                    options={['день', 'ночь', 'утро', 'вечер']}
                    availableValues={['ночь', 'утро', 'вечер']}
                    values={values}
                    onChange={handleChange}
                />
            )}
        </DeviceTypeContext.Consumer>
    );
};

Simple.parameters = {
    info: {
        title: 'Простой вариант с простыми опциями-строками',
    },
};

interface IObjectOption {
    id: string;
    text: string;
    price: number;
    comment?: string;
}

export const ObjectOptions: TStorybookFunctionComponent = () => {
    const [values, setValues] = useState<string[]>(['day']);

    const handleChange = useCallback(
        (option: IObjectOption, checked: boolean) => {
            const newValues = checked
                ? [...values, option.id]
                : values.filter(value => value !== option.id);

            setValues(newValues);
        },
        [values],
    );

    return (
        <DeviceTypeContext.Consumer>
            {(deviceType): ReactNode => (
                <div style={{width: 300}}>
                    <FilterCheckList
                        deviceType={deviceType}
                        options={[
                            {
                                id: 'day',
                                text: 'день',
                                price: 100,
                                comment: '12:00-16:00',
                            },
                            {
                                id: 'night',
                                text: 'ночь',
                                price: 10,
                                comment: '00:00-06:00',
                            },
                            {id: 'morning', text: 'утро', price: 10},
                            {
                                id: 'evening',
                                text: 'вечер',
                                price: 110,
                                comment: '00:00-06:00',
                            },
                        ]}
                        availableValues={['night', 'morning', 'evening']}
                        values={values}
                        onChange={handleChange}
                        renderText={(o): ReactNode => (
                            <div style={{width: 70}}>{o.text}</div>
                        )}
                        renderComment={(o): string | null => o.comment || null}
                        renderAside={(o): ReactNode => (
                            <Price
                                value={o.price}
                                currency={CurrencyType.RUB}
                                isFrom
                            />
                        )}
                        optionToValue={(o): string => o.id}
                    />
                </div>
            )}
        </DeviceTypeContext.Consumer>
    );
};

ObjectOptions.parameters = {
    info: {
        title: 'Вариант со сложными опциями (объектами), с комментом и ценой',
    },
};
