import React, {
    FunctionComponent,
    RefObject,
    useCallback,
    useMemo,
    useState,
} from 'react';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';

import Text from 'components/Text/Text';
import Box from 'components/Box/Box';
import CloseCircleIcon from 'icons/24/CloseCircle';
import ArrowRightIcon from 'icons/16/ArrowRight';
import FiltersBottomSheet from 'components/FiltersBottomSheet/FiltersBottomSheet';

import cx from './FilterNestedBottomSheet.scss';

const ONE_DIGIT_NUM_THRESHOLD = 10;

export interface IFilterInBottomSheetProps extends IWithQaAttributes {
    label: string;
    children: React.ReactNode;
    selectedCount?: number;
    onReset?: () => void;
    onClose?: () => void;

    firstBottomSheetRef: RefObject<HTMLDivElement>;
    withReset?: boolean;
    withArrow?: boolean;

    submitButtonLabel: string;
    submitButtonDisabled: boolean;

    /**
     * серый текст справа
     * @default undefined
     */
    subLabel?: string;
}

const FilterNestedBottomSheet: FunctionComponent<IFilterInBottomSheetProps> = ({
    label,
    children,
    selectedCount = 0,
    onReset,
    firstBottomSheetRef,
    submitButtonDisabled,
    submitButtonLabel,
    withArrow,
    withReset = true,
    subLabel,
    onClose,
    ...rest
}) => {
    const handleInlineReset = useCallback(
        (event: React.MouseEvent) => {
            event.preventDefault();
            event.stopPropagation();
            onReset?.();
        },
        [onReset],
    );
    const {
        setTrue: setOpenedStatus,
        setFalse: close,
        value: isOpened,
    } = useBoolean(false);

    const handleClose = useCallback(() => {
        close();

        onClose?.();
    }, [onClose, close]);

    const [maxSize, setMaxSize] = useState<string | undefined>();
    const getMaxSize = useCallback(() => {
        const parentHeight =
            firstBottomSheetRef.current?.parentElement?.offsetHeight;

        if (!parentHeight) {
            return;
        }

        return `calc(${parentHeight}px - 32px)`;
    }, [firstBottomSheetRef]);
    const open = useCallback(() => {
        setMaxSize(getMaxSize());
        setOpenedStatus();
    }, [getMaxSize, setOpenedStatus]);
    const buttonResetDisabled = selectedCount === 0;

    const countBadge = useMemo(
        () =>
            selectedCount > 0 && (
                <>
                    <div
                        className={cx(
                            'selectedCount',
                            selectedCount < ONE_DIGIT_NUM_THRESHOLD
                                ? 'selectedCount_len_1'
                                : 'selectedCount_len_2',
                        )}
                    >
                        {selectedCount}
                    </div>
                    <CloseCircleIcon
                        className={cx('inlineButtonReset')}
                        onClick={handleInlineReset}
                    />
                </>
            ),
        [selectedCount, handleInlineReset],
    );

    const arrow = useMemo(() => {
        if (countBadge || !withArrow) {
            return null;
        }

        return <ArrowRightIcon className={cx('arrow')} />;
    }, [countBadge, withArrow]);

    const subLabelNode = useMemo(() => {
        if (!subLabel) {
            return null;
        }

        return (
            <div className={cx('subLabel')}>
                <Text color="secondary">{subLabel}</Text>
            </div>
        );
    }, [subLabel]);

    return (
        <div>
            <button
                className={cx('previewButton')}
                type="button"
                onClick={open}
                {...prepareQaAttributes(rest)}
            >
                <Text>{label}</Text>
                {countBadge}
                {subLabelNode}
                {arrow}
            </button>
            <FiltersBottomSheet
                isOpened={isOpened}
                onClose={handleClose}
                maxSize={maxSize}
                header={label}
                resetButtonDisabled={buttonResetDisabled}
                onReset={onReset}
                submitButtonDisabled={submitButtonDisabled}
                onSubmit={close}
                submitLabel={submitButtonLabel}
                showResetButton={withReset}
            >
                <Box className={cx('scrollableContent')} between={8}>
                    {children}
                </Box>
            </FiltersBottomSheet>
        </div>
    );
};

export default React.memo(FilterNestedBottomSheet);
