import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/common-filters';

import Box from 'components/Box/Box';
import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import FiltersDesktopSkeleton from 'components/FiltersDesktop/components/FilterDesktopSkeleton/FiltersDesktopSkeleton';
import CloseIcon from 'icons/24/Close';

import cx from './FiltersDesktop.scss';

export interface IFiltersDesktopProps
    extends IWithClassName,
        IWithQaAttributes {
    loading?: boolean;
    skeletonsWidth: number[];
    children: React.ReactNode;
    rightNode?: React.ReactNode;
    canReset: boolean;
    onReset?: () => void;
    resetButtonType?: 'text' | 'icon';
    withShadow?: boolean;
    withoutHorizontalPaddings?: boolean;
}

const FiltersDesktop: React.FC<IFiltersDesktopProps> = ({
    className,
    loading = false,
    skeletonsWidth,
    canReset,
    children,
    rightNode,
    onReset,
    resetButtonType = 'text',
    withShadow = true,
    withoutHorizontalPaddings = false,
    ...rest
}) => {
    const buttonContent =
        resetButtonType === 'text' ? i18nBlock.reset() : <CloseIcon />;

    return (
        <div className={cx('root', {withShadow})}>
            <Box
                className={cx('wrapper', {withoutHorizontalPaddings})}
                between={2}
                y={3}
                inline
                nowrap
                {...prepareQaAttributes(rest)}
            >
                <Flex
                    className={cx('items', {items_loading: loading}, className)}
                    inline
                    flexWrap="wrap"
                >
                    {loading ? (
                        <FiltersDesktopSkeleton
                            skeletonsWidth={skeletonsWidth}
                        />
                    ) : (
                        <>
                            {children}
                            {canReset && (
                                <Button
                                    theme="attention"
                                    onClick={onReset}
                                    className={cx({
                                        buttonIcon: resetButtonType === 'icon',
                                    })}
                                    {...prepareQaAttributes({
                                        parent: rest,
                                        current: 'reset',
                                    })}
                                >
                                    {buttonContent}
                                </Button>
                            )}
                        </>
                    )}
                </Flex>

                {rightNode}
            </Box>
        </div>
    );
};

export default React.memo(FiltersDesktop);
