import React from 'react';
import {partition} from 'lodash';

import getMenuItems from 'components/Footer/constants/footerMenuItems';
import serviceCrossLinks from 'components/Footer/constants/serviceCrossLinks';
import {EProjectName} from 'constants/common';
import {PARTNERS_FOOTER_MENU_ITEMS} from 'components/Footer/constants/partnersFooterMenuItems';

import {IFooterCommonProps} from 'components/Footer/types';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';

import * as i18nBlock from 'i18n/components';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import FooterBlock from 'components/Footer/components/FooterBlock/FooterBlock';
import FooterCopyright from 'components/Footer/components/FooterCopyright/FooterCopyright';
import FooterLinks from 'components/Footer/components/FooterLinks/FooterLinks';
import SocialLinks from 'components/Footer/components/SocialLinks/SocialLinks';
import AppLinks from 'components/Footer/components/AppLinks/AppLinks';
import SupportPhone from 'components/SupportPhone/SupportPhone';

const IndexFooter: React.FC<IFooterCommonProps> = props => {
    const {project, footerType} = props;
    const deviceType = useDeviceType();
    const {isWhiteLabel, config: whiteLabelConfig} = useWhiteLabelConfig();
    const whiteLabelFooterBlock = whiteLabelConfig?.footerBlock;

    const [partnersLinks, userLinks] = partition(
        getMenuItems({footerType, withPlus: !isWhiteLabel}),
        item => PARTNERS_FOOTER_MENU_ITEMS.has(item),
    );

    const serviceCrossLinksBlock = (
        <FooterBlock title={i18nBlock.footerServiceCrossLinksTitle()}>
            <FooterLinks type="vertical" links={serviceCrossLinks} />
        </FooterBlock>
    );

    const userLinksBlock = (
        <FooterBlock title={i18nBlock.footerUsersLinksTitle()}>
            <FooterLinks
                type={deviceType.isDesktop ? 'vertical' : 'horizontal'}
                links={userLinks}
            />
        </FooterBlock>
    );

    const partnersLinksBlock = (
        <FooterBlock title={i18nBlock.footerPartnersLinksTitle()}>
            <FooterLinks
                type={deviceType.isDesktop ? 'vertical' : 'horizontal'}
                links={partnersLinks}
            />
        </FooterBlock>
    );

    const whiteLabelLinksBlock = whiteLabelFooterBlock ? (
        <FooterBlock title={whiteLabelFooterBlock.title}>
            <FooterLinks
                type={deviceType.isDesktop ? 'vertical' : 'horizontal'}
                links={whiteLabelFooterBlock.links.map(({name, link}) => ({
                    name,
                    link: encodeURI(link),
                }))}
            />
        </FooterBlock>
    ) : null;

    const supportPhoneBlock =
        project === EProjectName.BUSES ? (
            <FooterBlock title={i18nBlock.footerSupportTitle()}>
                <Text size="s">
                    <SupportPhone />
                </Text>
            </FooterBlock>
        ) : null;

    const socialBlock = (
        <FooterBlock title={i18nBlock.footerSocialsTitle()}>
            <SocialLinks />
        </FooterBlock>
    );

    const appLinkBlock = (
        <FooterBlock title={i18nBlock.footerAviaAppTitle()}>
            <AppLinks />
        </FooterBlock>
    );

    const copyrightBlock = <FooterCopyright />;

    if (deviceType.isMobile) {
        return (
            <Flex flexDirection="column" between={6}>
                {!isWhiteLabel && serviceCrossLinksBlock}
                {userLinksBlock}
                {partnersLinksBlock}
                {isWhiteLabel && whiteLabelLinksBlock}
                {!isWhiteLabel && supportPhoneBlock}
                {!isWhiteLabel && socialBlock}
                {!isWhiteLabel && appLinkBlock}

                {copyrightBlock}
            </Flex>
        );
    }

    return (
        <Flex flexDirection="column" between={7}>
            <Flex flexDirection="row" justifyContent="space-between">
                {!isWhiteLabel && serviceCrossLinksBlock}
                {userLinksBlock}
                {partnersLinksBlock}
                {isWhiteLabel && whiteLabelLinksBlock}
                <Flex flexDirection="column" between={7}>
                    {!isWhiteLabel && supportPhoneBlock}
                    {!isWhiteLabel && socialBlock}
                    {!isWhiteLabel && appLinkBlock}
                </Flex>
            </Flex>

            {copyrightBlock}
        </Flex>
    );
};

export default IndexFooter;
