import {Component, ReactNode} from 'react';
import {Form as FinalForm, FormProps} from 'react-final-form';
import {FormApi, ValidationErrors, AnyObject} from 'final-form';
import _isEmpty from 'lodash/isEmpty';

import {IFormValidationInfo} from 'types/common/validation/form';

import validateValues from 'utilities/validation/form/validateValues';
import mergeErrors from './utilities/mergeErrors';

import FieldGroup, {
    FieldGroupContext,
} from './components/FieldGroup/FieldGroup';
import Field from './components/Field/Field';

export interface IFormProps<FormValues> extends FormProps<FormValues> {
    validationInfo?: IFormValidationInfo;
    /** Callback на submit, для кастомной валидации */
    onSubmitValidation?: FormProps<FormValues>['onSubmit'];
}

class Form<FormValues = AnyObject> extends Component<IFormProps<FormValues>> {
    static Field = Field;

    static FieldGroup = FieldGroup;

    static FieldGroupContext = FieldGroupContext;

    /* Helpers */

    private validate = (values: FormValues): ValidationErrors => {
        const {validationInfo, validate: customValidate} = this.props;
        let errors: ValidationErrors = {};

        if (validationInfo) {
            errors = validateValues(validationInfo, values);
        }

        if (customValidate) {
            const customErrors = customValidate(values);

            mergeErrors(errors, customErrors);
        }

        return errors;
    };

    private submitHandler = (
        values: FormValues,
        form: FormApi<FormValues>,
    ): ValidationErrors => {
        const {validationInfo, onSubmitValidation, onSubmit} = this.props;
        let errors: ValidationErrors = {};

        // Валидация по общей спеке
        if (validationInfo) {
            errors = validateValues(validationInfo, values);
        }

        // Кастомная валидация
        if (onSubmitValidation) {
            const customErrors = onSubmitValidation(values, form);

            mergeErrors(errors, customErrors);
        }

        if (_isEmpty(errors)) {
            if (onSubmit) {
                const submitErrors = onSubmit(values, form);

                mergeErrors(errors, submitErrors);
            }
        }

        return errors;
    };

    /* Render */

    render(): ReactNode {
        const {validationInfo, validate, onSubmit, ...props} = this.props;

        return (
            <FinalForm<FormValues>
                validate={this.validate}
                onSubmit={this.submitHandler}
                {...props}
            />
        );
    }
}

export {useFieldValue} from './hooks/useFieldValue';

export default Form;
