import React, {ReactNode} from 'react';
import {FieldValidator} from 'final-form';
import {FieldMetaState} from 'react-final-form';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {TValidator} from 'types/common/validation/form';

import CommonFormField, {
    IFieldProps as ICommonFormFieldProps,
} from 'components/Form/components/Field/Field';
import Field, {IFieldProps} from './components/Field/Field';

export interface IFormFieldProps<
    TChangeEvent,
    TFieldValue = unknown,
    TAllValues extends object = object,
> extends Omit<IFieldProps<TChangeEvent>, 'fieldProps'>,
        Omit<ICommonFormFieldProps, 'validate'>,
        IWithClassName,
        IWithDeviceType {
    isHiddenError?: boolean;
    children?: never;
    inputRef?: (input: HTMLInputElement | null) => void;
    validate?: TValidator<TFieldValue, TAllValues>;

    onChange?: (event: TChangeEvent) => void;
    onFocus?: (event?: React.FocusEvent<HTMLElement>) => void;
    onBlur?: (event?: React.FocusEvent<HTMLElement>) => void;
    onChangeError?: (fieldName: string, meta: FieldMetaState<string>) => void;
}

function FormField<
    TChangeEvent,
    TFieldValue = unknown,
    TAllValues extends object = object,
>({
    id,
    name,
    control,
    title,
    className,
    hint,
    isHiddenError,
    deviceType,
    inputRef,
    onChange,
    onFocus,
    onBlur,
    showErrorsInTooltip,
    validate,
    onChangeError,
    ...formFieldProps
}: IFormFieldProps<TChangeEvent, TFieldValue, TAllValues>): React.ReactElement {
    return (
        <CommonFormField
            name={name}
            // Костыль из-за того, что в react-final-form не предусмотрена типизация значений всей формы
            // https://github.com/final-form/final-form/blob/0d7aecbc37c140610361506781fe839b85302836/src/index.d.ts#L133
            validate={validate as FieldValidator<any>}
            {...formFieldProps}
        >
            {(fieldProps): ReactNode => (
                <Field
                    id={id}
                    control={control}
                    title={title}
                    className={className}
                    hint={hint}
                    isHiddenError={isHiddenError}
                    deviceType={deviceType}
                    fieldProps={fieldProps}
                    onChange={onChange}
                    onFocus={onFocus}
                    onBlur={onBlur}
                    inputRef={inputRef}
                    isTooltipErrorOnMobile={showErrorsInTooltip}
                    onChangeError={onChangeError}
                />
            )}
        </CommonFormField>
    );
}

export default FormField;
