import React, {useEffect, useCallback, useState, useRef, useMemo} from 'react';
import {FieldMetaState, FieldRenderProps} from 'react-final-form';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {usePrevious} from 'utilities/hooks/usePrevious';
import getValidationErrorMessage from '../../utilities/getValidationErrorMessage';

import Hint from 'components/Hint/Hint';

import Tooltip from '../Tooltip/Tooltip';

import cx from './Field.scss';

export interface IControlRenderProps {
    error?: string;
    controlRef: React.RefObject<HTMLDivElement>;
    inputRef: React.MutableRefObject<HTMLInputElement | null>;
}

export type TControlRenderFunc<TFieldValue = any> = (
    fieldProps: FieldRenderProps<TFieldValue, HTMLElement>,
    controlProps: IControlRenderProps,
) => React.ReactNode;

export interface IFieldProps<TChangeEvent>
    extends IWithClassName,
        IWithDeviceType {
    id?: string;
    title?: React.ReactNode;
    hint?: string;
    isHiddenError?: boolean;
    control: TControlRenderFunc;
    fieldProps: FieldRenderProps<any, HTMLElement>;
    isTooltipErrorOnMobile?: boolean;

    onChange?: (event: TChangeEvent) => void;
    onFocus?: (event?: React.FocusEvent<HTMLElement>) => void;
    onBlur?: (event?: React.FocusEvent<HTMLElement>) => void;
    onChangeError?: (fieldName: string, meta: FieldMetaState<string>) => void;

    inputRef?: (input: HTMLInputElement | null) => void;
}

function Field<TChangeEvent>({
    className,
    id,
    title,
    hint,
    isHiddenError,
    control,
    deviceType: {isDesktop, isMobile},
    fieldProps: {input, meta, ...restFieldProps},
    isTooltipErrorOnMobile,
    onChange,
    onFocus,
    onBlur,
    onChangeError,
    inputRef: propsInputRef,
}: IFieldProps<TChangeEvent>): React.ReactElement {
    const {name} = input;
    const error = useMemo(() => {
        return isHiddenError
            ? undefined
            : getValidationErrorMessage(input.value, meta);
    }, [input.value, meta, isHiddenError]);

    /* Хак чтобы во время анимации исчезновения, контент тултипа не был пустым */
    const lastError = usePrevious(error);

    const controlRef = useRef<HTMLDivElement | null>(null);
    const inputRef = useRef<HTMLInputElement | null>(null);
    const [isFocused, setFocused] = useState<boolean>(false);
    const [isHovered, setHovered] = useState<boolean>(false);

    const handleMouseEnter = useCallback(() => {
        setHovered(true);
    }, [setHovered]);

    const handleMouseLeave = useCallback(() => {
        setHovered(false);
    }, [setHovered]);

    const handleFocus = useCallback(
        (event: React.FocusEvent<HTMLElement> | undefined) => {
            if (onFocus) {
                onFocus(event);
            }

            input.onFocus(event);
            setFocused(true);
        },
        [input, setFocused, onFocus],
    );

    const handleBlur = useCallback(
        (event: React.FocusEvent<HTMLElement> | undefined) => {
            if (onBlur) {
                onBlur(event);
            }

            input.onBlur(event);
            setFocused(false);
        },
        [input, setFocused, onBlur],
    );

    const handleChange = useCallback(
        (event: TChangeEvent) => {
            input.onChange(event);

            if (onChange) {
                onChange(event);
            }
        },
        [input, onChange],
    );

    const controlFieldProps = useMemo(
        () => ({
            meta: meta,
            input: {
                ...input,
                ...restFieldProps,
                onFocus: handleFocus,
                onBlur: handleBlur,
                onChange: handleChange,
            },
        }),
        [handleFocus, handleBlur, handleChange, input, meta, restFieldProps],
    );

    /*
     * Совместимость для callbackRef и стандартного ref
     * Выпилить если только отпадет необходимость в `inputRef?: (input: HTMLInputElement | null) => void` в лего
     * И будет работать обычный RefObject.
     */
    useEffect(() => {
        if (propsInputRef) {
            propsInputRef(inputRef.current);
        }
    }, [propsInputRef, inputRef]);

    /*  */
    useEffect(() => {
        if (error && onChangeError) {
            onChangeError(name, meta);
        }
    }, [error]);

    /*
        Специфический момент final-form.
        Методы FormApi для фокуса Field по сути выставяляют состояние active
    */
    useEffect(() => {
        if (controlFieldProps.meta.active && inputRef?.current) {
            inputRef.current.focus();
        }
    }, [controlFieldProps.meta.active]);

    // Хак, нужен из-за того, что ЯБро для автокомплита смотрит не только
    // на атрибуты поля, но и на текст лэйбла к нему.
    // Чтобы смягчить должен быть выставлен area-label элементам с этой строкой
    const fixedTitle =
        title && typeof title === 'string' ? (
            <>
                {title.slice(0, 1)}
                <span className={cx('autocompleteFix')}>1</span>
                {title.slice(1)}
            </>
        ) : (
            title
        );

    const isHintError = isMobile && !isTooltipErrorOnMobile;
    const isTooltipError = isDesktop || isTooltipErrorOnMobile;

    const hintError = isHintError && error;

    return (
        <>
            <div
                className={cx('root', className)}
                onMouseEnter={handleMouseEnter}
                onMouseLeave={handleMouseLeave}
            >
                {title && (
                    <div className={cx('title')}>
                        <label htmlFor={id || name} area-label={title}>
                            {fixedTitle}
                        </label>
                    </div>
                )}
                <div className={cx('control')}>
                    <Hint
                        className={cx('hint')}
                        message={hintError || hint}
                        isError={Boolean(hintError)}
                        isAnimated={!isMobile}
                    >
                        {control(controlFieldProps, {
                            error,
                            controlRef,
                            inputRef,
                        })}
                    </Hint>
                </div>
            </div>
            {isTooltipError && controlRef && (
                <Tooltip
                    isVisible={Boolean(error) && (isHovered || isFocused)}
                    anchorRef={controlRef}
                    content={error || lastError}
                />
            )}
        </>
    );
}

export default Field;
