import {useCallback, useEffect, useRef, useState} from 'react';

const THRESHOLD = 20;
const VISIBLE_AREA = 200;

interface IScrollHidingProps {
    threshold?: number;
    visibleArea?: number;
}

export function useScrollHiding(props: IScrollHidingProps = {}): boolean {
    const [isHidden, setHidden] = useState(false);
    const lastScrollY = useRef(0);

    const {threshold = THRESHOLD, visibleArea = VISIBLE_AREA} = props;

    const handleScroll = useCallback(() => {
        if (window.scrollY < visibleArea) {
            setHidden(false);

            return;
        }

        const diff = lastScrollY.current - window.scrollY;

        if (diff > threshold) {
            lastScrollY.current = window.scrollY;
            setHidden(false);
        } else if (diff < -threshold) {
            lastScrollY.current = window.scrollY;
            setHidden(true);
        }
    }, [setHidden, threshold, visibleArea]);

    useEffect(() => {
        lastScrollY.current = window.scrollY;
        window.addEventListener('scroll', handleScroll);

        return (): void => {
            window.removeEventListener('scroll', handleScroll);
        };
    }, [handleScroll]);

    return isHidden;
}
