/* eslint-disable comma-dangle */
import React, {PureComponent, ReactNode} from 'react';
import throttle from 'lodash/throttle';
import debounce from 'lodash/debounce';

import {EProjectName} from 'constants/common';

import {IWithDeviceType} from 'types/withDeviceType';
import {IWithWhiteLabelConfig} from 'types/common/whiteLabel/IWithWhiteLabelConfig';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {TExperiments} from 'server/providers/experiments/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {loadable} from 'utilities/componentLoadable';
import {isElementMounted} from 'utilities/dom/isElementMounted';
import {isElementInsidePopup} from 'utilities/dom/isElementInsidePopup';
import {createEvent} from 'utilities/functions/createEvent';
import {reachGoal} from 'utilities/metrika';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import {IHeaderDeviceBase} from 'components/Header/Header';
import YandexTravelLogo from 'components/YandexTravelLogo/YandexTravelLogo';
import NavigationContainer from 'components/NavigationContainer/NavigationContainer';
import WhiteLabelLogo from 'components/Header/components/WhiteLabelLogo/WhiteLabelLogo';
import UserInfo from '../../components/UserInfo/UserInfo';
import {IBaseSearchFormProps} from 'components/SearchForm/SearchForm';

import ScopeContext from 'contexts/ScopeContext';

import cx from './HeaderDesktop.scss';

/* SearchForm */
const AviaSearchForm = loadable(
    () => import('projects/avia/components/SearchForm/SearchForm'),
);
const TrainsSearchForm = loadable(
    () => import('projects/trains/components/SearchForm/SearchForm'),
);
const HotelsSearchForm = loadable(
    () => import('projects/hotels/components/SearchForm/SearchForm'),
);
const BusesSearchForm = loadable(
    () => import('projects/buses/components/SearchForm/SearchForm'),
);

/* SearchInformation */
const AviaSearchInformation = loadable(
    () =>
        import(
            'projects/avia/containers/AviaSearchInformation/AviaSearchInformation'
        ),
);
const TrainsSearchInformation = loadable(
    () => import('containers/TrainsSearchInformation/TrainsSearchInformation'),
);
const HotelsSearchInformation = loadable(
    () =>
        import(
            'projects/hotels/containers/HotelsSearchInformation/HotelsSearchInformation'
        ),
);
const BusesSearchInformation = loadable(
    () =>
        import('projects/buses/components/SearchInformation/SearchInformation'),
);

const THROTTLE_DELAY = 100;
/**
 * Таймаут, для функции, которая меняет флаг при взаимодействии с формой.
 * При этом значении не происходит смены флага с true на false и обратно в true,
 * когда мы переключаемся по элементам формы.
 * Выяснен эмпирическим путем.
 *
 * @see https://st.yandex-team.ru/TRAVELFRONT-968
 */
const INTERACTION_DEBOUNCE_DELAY = 200;
const SAFE_ZONE_HEIGHT = 100;
const DELTA_THRESHOLD = 160;

interface IHeaderDesktopPropTypes
    extends IHeaderDeviceBase,
        IWithDeviceType,
        IWithWhiteLabelConfig {
    /**
     * шапка зафиксирована в одном состоянии, котрое определяется searchFormInitialIsExpanded
     */
    searchFormIsStatic?: boolean;
    initiallyCalendarIsOpen?: boolean;
    searchFormInitialIsExpanded?: boolean;
    experiments: TExperiments;
    hideSearchInformation?: boolean;
    borderBottomType: EHeaderBorderBottomType;
    withYandexLogo?: boolean;
    searchFormProps?: IBaseSearchFormProps;
}

interface IHeaderDesktopState {
    searchFormIsExpanded: boolean;
    searchFormIsLocked: boolean;
    refIsSet: boolean;
}

class HeaderDesktop extends PureComponent<
    IHeaderDesktopPropTypes,
    IHeaderDesktopState
> {
    readonly state: IHeaderDesktopState = {
        searchFormIsExpanded: Boolean(this.props.searchFormInitialIsExpanded),
        searchFormIsLocked: false,
        refIsSet: false,
    };

    private delta: number = 0;

    private previousScrollTop: number = 0;

    private readonly headerRef = React.createRef<HTMLElement>();
    private containerRef: HTMLDivElement | null = null;

    componentDidMount(): void {
        const {searchFormIsAvailable, searchFormIsStatic} = this.props;

        if (searchFormIsAvailable && !searchFormIsStatic) {
            this.previousScrollTop = window.pageYOffset;
            window.addEventListener('scroll', this.onScroll);
            document.addEventListener('mousedown', this.onDocumentClick);
        }
    }

    componentDidUpdate(
        prevProps: Readonly<IHeaderDesktopPropTypes>,
        prevState: Readonly<IHeaderDesktopState>,
    ): void {
        const {searchFormIsExpanded} = this.state;

        // Если переходим по браузерным табам, то в это время происходит скрытие шапки.
        // После перехода в таб нашего приложения шапка снова открывается. И открывается попап у саджеста.
        // Этот попап позиционируется неправильно, потому позиционирование происходит во время анимации.
        // Обходим этот момент хаком, описанным в lego.
        if (
            prevState.searchFormIsExpanded !== searchFormIsExpanded &&
            searchFormIsExpanded
        ) {
            setTimeout(() => {
                document.dispatchEvent(createEvent('documentchange'));
            }, 300);
        }

        if (
            this.props.searchFormInitialIsExpanded !==
                prevProps.searchFormInitialIsExpanded &&
            this.props.searchFormInitialIsExpanded !==
                this.state.searchFormIsExpanded
        ) {
            this.setState({
                searchFormIsExpanded: Boolean(
                    this.props.searchFormInitialIsExpanded,
                ),
            });
        }
    }

    componentWillUnmount(): void {
        window.removeEventListener('scroll', this.onScroll);
        document.removeEventListener('mousedown', this.onDocumentClick);
    }

    private onSubmitForm = (): void => {
        const {searchFormInitialIsExpanded, onSubmitForm} = this.props;

        this.setState({
            searchFormIsExpanded: Boolean(searchFormInitialIsExpanded),
            searchFormIsLocked: false,
        });

        onSubmitForm?.();
    };

    private onCollapsedSearchFormClick = (): void => {
        this.delta = 0;
        this.setState({
            searchFormIsExpanded: true,
        });

        reachGoal(ECommonGoal.HEADER_SEARCH_FORM_INFORMATION_CLICK);
    };

    private onInteractWithSearchForm = debounce((isInteracting: boolean) => {
        this.setState({
            searchFormIsLocked: isInteracting,
        });
    }, INTERACTION_DEBOUNCE_DELAY);

    // Обрабатываем клик за пределами шапки
    private onDocumentClick = (event: MouseEvent): void => {
        const headerNode: HTMLElement | null = this.headerRef.current;
        const targetNode = event.target as HTMLElement;

        // Календарь после клика тут же скрывается и анмаунтится.
        // Чтобы после клика по дате в календаре форма не сворачивалась
        // необходимо проверять, что targetNode находится в DOM дереве.
        if (!headerNode || !isElementMounted(targetNode)) {
            return;
        }

        if (
            !headerNode.contains(targetNode) &&
            !isElementInsidePopup(targetNode)
        ) {
            this.setState({
                searchFormIsExpanded: false,
            });
        }
    };

    private onScroll = throttle(
        () => {
            const {searchFormIsLocked} = this.state;
            const {searchFormInitialIsExpanded} = this.props;

            const scrollTop = window.pageYOffset;

            if (scrollTop < SAFE_ZONE_HEIGHT) {
                if (!searchFormIsLocked) {
                    // Раскрываем/скрываем форму когда приближаемся к верху страницы
                    // и ставим delta в пороговое значение, чтобы после преодоления безопасной зоны произошло сворачивание/раскрытие формы
                    this.delta = DELTA_THRESHOLD;
                    this.setState({
                        searchFormIsExpanded: Boolean(
                            searchFormInitialIsExpanded,
                        ),
                    });
                }
            } else {
                const scrollOffset = scrollTop - this.previousScrollTop;
                const newDelta = this.delta + scrollOffset;

                // Считаем разницу проскролленного в одну из сторон (вверх или вниз)
                // Если направление скролла было изменено, то сбрасываем разницу на величину scrollOffset
                this.delta =
                    Math.abs(newDelta) > Math.abs(this.delta)
                        ? newDelta
                        : scrollOffset;

                // Скрываем форму только если было преодолено определенное значение и скроллили вниз
                // Если нет взаимодействия с формой, то можно ее скрыть
                if (this.delta >= DELTA_THRESHOLD && !searchFormIsLocked) {
                    this.setState({searchFormIsExpanded: false});
                }
            }

            // Перестраховывемся на случай мобильных устройств, которые могут скроллить за пределы страницы
            this.previousScrollTop = scrollTop < 0 ? 0 : scrollTop;
        },
        THROTTLE_DELAY,
        {trailing: true},
    );

    private setContainerRef = (node: HTMLDivElement | null): void => {
        this.containerRef = node;

        // Необходимо чтобы зафорсить обновление потомков на изменение ref'а
        // До этого работало т.к. не было статичного отображения формы
        // и необходимый апдейт происходил при разворачивании формы
        this.setState({
            refIsSet: node !== null,
        });
    };

    private renderSearchInformationByProject(): ReactNode {
        const {project, deviceType, hideSearchInformation} = this.props;

        if (hideSearchInformation) return null;

        switch (project) {
            case EProjectName.TRAINS:
                return (
                    <TrainsSearchInformation
                        className={cx('searchFormCollapsed')}
                        onClick={this.onCollapsedSearchFormClick}
                        deviceType={deviceType}
                    />
                );
            case EProjectName.AVIA:
                return (
                    <AviaSearchInformation
                        className={cx('searchFormCollapsed')}
                        onClick={this.onCollapsedSearchFormClick}
                    />
                );
            case EProjectName.HOTELS: {
                return (
                    <HotelsSearchInformation
                        className={cx('searchFormCollapsed')}
                        onClick={this.onCollapsedSearchFormClick}
                        deviceType={deviceType}
                    />
                );
            }
            case EProjectName.BUSES:
                return (
                    <BusesSearchInformation
                        className={cx('searchFormCollapsed')}
                        onClick={this.onCollapsedSearchFormClick}
                    />
                );

            default:
                return null;
        }
    }

    private renderSearchFormByProject(): ReactNode {
        const {searchFormIsExpanded} = this.state;
        const {
            project,
            initiallyCalendarIsOpen,
            validateSearchFormOnMount,
            hideSearchInformation,
            searchFormProps,
        } = this.props;

        if (hideSearchInformation) {
            return null;
        }

        switch (project) {
            case EProjectName.TRAINS:
                return (
                    <TrainsSearchForm
                        className={cx('searchForm')}
                        onSubmit={this.onSubmitForm}
                        onInteract={this.onInteractWithSearchForm}
                        isExpanded={searchFormIsExpanded}
                        initiallyCalendarIsOpen={initiallyCalendarIsOpen}
                        {...searchFormProps}
                    />
                );
            case EProjectName.AVIA:
                return (
                    <AviaSearchForm
                        className={cx('searchForm')}
                        onSubmit={this.onSubmitForm}
                        onInteract={this.onInteractWithSearchForm}
                        isExpanded={searchFormIsExpanded}
                        useFilters
                        validateOnMount={validateSearchFormOnMount}
                        initiallyCalendarIsOpen={initiallyCalendarIsOpen}
                        {...searchFormProps}
                    />
                );
            case EProjectName.HOTELS: {
                return (
                    <HotelsSearchForm
                        className={cx('searchForm')}
                        hasDelayBeforeUpdateLocation
                        isExpanded={searchFormIsExpanded}
                        onSubmit={this.onSubmitForm}
                        onInteract={this.onInteractWithSearchForm}
                        {...searchFormProps}
                    />
                );
            }
            case EProjectName.BUSES:
                return (
                    <BusesSearchForm
                        className={cx('searchForm')}
                        onSubmit={this.onSubmitForm}
                        onInteract={this.onInteractWithSearchForm}
                        isExpanded={searchFormIsExpanded}
                        {...searchFormProps}
                    />
                );

            default:
                return null;
        }
    }

    render(): ReactNode {
        const {searchFormIsExpanded} = this.state;
        const {
            className,
            containerClassName,
            wrapperClassName,
            formWrapperClassName,
            isFixed,
            project,
            showNavigation,
            searchFormIsStatic,
            searchFormInitialIsExpanded,
            searchFormIsAvailable,
            borderBottomType,
            withYandexLogo,
            withNavCaptions,
            whiteLabelConfig,
        } = this.props;

        const withBorderBottom = project !== EProjectName.AVIA;

        const searchInformationNode =
            searchFormIsAvailable && this.renderSearchInformationByProject();

        const isWhiteLabel = Boolean(whiteLabelConfig);

        return (
            <header
                {...prepareQaAttributes(this.props)}
                ref={this.headerRef}
                className={cx(
                    'root',
                    {
                        root_fixed: isFixed,
                        root_static: searchFormIsStatic,
                        root_form: searchFormIsAvailable,
                        root_formIsExpanded:
                            searchFormIsAvailable && searchFormIsExpanded,
                        root_formInitialIsExpanded:
                            searchFormIsAvailable &&
                            searchFormInitialIsExpanded,
                    },
                    className,
                    `root_border_${borderBottomType}`,
                )}
            >
                <div
                    className={cx('container', containerClassName)}
                    ref={this.setContainerRef}
                >
                    <div className={cx('content')}>
                        <div className={cx('wrapper', wrapperClassName)}>
                            <div className={cx('main')}>
                                {isWhiteLabel ? (
                                    <WhiteLabelLogo project={project} />
                                ) : (
                                    <YandexTravelLogo
                                        withYandexLogo={withYandexLogo}
                                        project={project}
                                        className={cx('logo')}
                                        {...prepareQaAttributes({
                                            parent: this.props,
                                            current: 'portalLogo',
                                        })}
                                    />
                                )}

                                {showNavigation && !isWhiteLabel && (
                                    <NavigationContainer
                                        className={cx('navigation')}
                                        collapsed={Boolean(
                                            searchInformationNode,
                                        )}
                                        {...prepareQaAttributes({
                                            parent: this.props,
                                            current: 'navigation',
                                        })}
                                    />
                                )}

                                {searchInformationNode}

                                <UserInfo
                                    className={cx('userInfo')}
                                    hasSideSheetNavigation={false}
                                    withNavCaptions={withNavCaptions}
                                    {...prepareQaAttributes({
                                        parent: this.props,
                                        current: 'userInfo',
                                    })}
                                />
                            </div>
                        </div>
                    </div>

                    {searchFormIsAvailable && (
                        <div className={cx('formContent', {withBorderBottom})}>
                            <div
                                className={cx(
                                    'wrapper',
                                    'wrapper_flexible',
                                    formWrapperClassName,
                                )}
                            >
                                <ScopeContext.Provider
                                    value={this.containerRef || undefined}
                                >
                                    {this.renderSearchFormByProject()}
                                </ScopeContext.Provider>
                            </div>
                        </div>
                    )}
                </div>
            </header>
        );
    }
}

export default React.memo(HeaderDesktop);
