import React, {useCallback, useMemo} from 'react';

import {EProjectName} from 'constants/common';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {loadable} from 'utilities/componentLoadable';
import {reachGoal} from 'utilities/metrika';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {getQa, prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Flex from 'components/Flex/Flex';
import Modal, {EModalAnimationType} from 'components/Modal/Modal';
import {IHeaderDeviceBase} from 'components/Header/Header';
import YandexTravelLogo from 'components/YandexTravelLogo/YandexTravelLogo';
import NavigationContainer from 'components/NavigationContainer/NavigationContainer';
import UserInfo from '../../components/UserInfo/UserInfo';
import WhiteLabelLogo from '../../components/WhiteLabelLogo/WhiteLabelLogo';
import {IBaseSearchFormProps} from 'components/SearchForm/SearchForm';

import cx from './HeaderMobile.scss';

const FORM_LOADABLE_FALLBACK_OPTIONS = {
    hasFallback: true,
    className: cx('formChunkLoader'),
};

/* SearchForm */
const TrainsSearchForm = loadable(
    () => import('projects/trains/components/SearchForm/SearchForm'),
    FORM_LOADABLE_FALLBACK_OPTIONS,
);
const AviaSearchForm = loadable(
    () => import('projects/avia/components/SearchForm/SearchForm'),
    FORM_LOADABLE_FALLBACK_OPTIONS,
);
const HotelsSearchForm = loadable(
    () => import('projects/hotels/components/SearchForm/SearchForm'),
    FORM_LOADABLE_FALLBACK_OPTIONS,
);
const BusesSearchForm = loadable(
    () => import('projects/buses/components/SearchForm/SearchForm'),
    FORM_LOADABLE_FALLBACK_OPTIONS,
);

/* SearchInformation */
const AviaSearchInformation = loadable(
    () =>
        import(
            'projects/avia/containers/AviaSearchInformation/AviaSearchInformation'
        ),
);
const TrainsSearchInformation = loadable(
    () => import('containers/TrainsSearchInformation/TrainsSearchInformation'),
);
const HotelsSearchInformation = loadable(
    () =>
        import(
            'projects/hotels/containers/HotelsSearchInformation/HotelsSearchInformation'
        ),
);
const BusesSearchInformation = loadable(
    () =>
        import('projects/buses/components/SearchInformation/SearchInformation'),
);

/* NavigationSideSheet */
const NavigationSideSheet = loadable(
    () => import('components/NavigationSideSheet/NavigationSideSheet'),
);

interface IHeaderMobileProps extends IHeaderDeviceBase {
    filters?: React.ReactNode;
    fastFilters?: React.ReactNode;
    hasSideSheetNavigation?: boolean;
    searchFormInitialIsExpanded?: boolean;
    hideTopMobileBlock?: boolean;
    searchFormCollapsedClassName?: string;
    hideSearchInformation?: boolean;
    borderBottomType: EHeaderBorderBottomType;
    showFavoritesLink?: boolean;
    searchFormProps?: IBaseSearchFormProps;
}

const SUPPORTED_PROJECTS = [
    EProjectName.AVIA,
    EProjectName.TRAINS,
    EProjectName.HOTELS,
    EProjectName.BUSES,
];

const HeaderMobile: React.FC<IHeaderMobileProps> = props => {
    const {
        className,
        searchFormCollapsedClassName,
        project,
        showNavigation,
        hasSideSheetNavigation,
        searchFormIsAvailable,
        validateSearchFormOnMount,
        filters,
        fastFilters,
        onSubmitForm,
        hideTopMobileBlock,
        hideSearchInformation,
        borderBottomType,
        withNavCaptions,
        showFavoritesLink,
        searchFormProps,
    } = props;

    const deviceType = useDeviceType();
    const {isWhiteLabel} = useWhiteLabelConfig();

    const {aviaNewTouchFilters} = useExperiments();

    const {
        value: modalIsVisible,
        setTrue: showModal,
        setFalse: hideModal,
    } = useBoolean(false);

    const portalLogoQa = getQa({
        parent: props,
        current: 'portalLogo',
    });

    const userInfoQa = getQa({
        parent: props,
        current: 'userInfo',
    });

    const navigationSideSheetQa = getQa({
        parent: props,
        current: 'navigationSideSheet',
    });

    const onCollapsedSearchFormClick = useCallback(() => {
        showModal();

        reachGoal(ECommonGoal.HEADER_SEARCH_FORM_INFORMATION_CLICK);
    }, [showModal]);

    const handleSubmitForm = useCallback(() => {
        onSubmitForm?.();

        hideModal();
    }, [hideModal, onSubmitForm]);

    const searchInformationBlock = useMemo(() => {
        if (hideSearchInformation) return null;

        switch (project) {
            case EProjectName.AVIA:
                return aviaNewTouchFilters ? null : (
                    <AviaSearchInformation
                        className={cx(
                            'searchFormCollapsed',
                            searchFormCollapsedClassName,
                        )}
                        onClick={onCollapsedSearchFormClick}
                    />
                );
            case EProjectName.TRAINS:
                return (
                    <Flex between={2} flexDirection="column">
                        <Flex
                            className={cx(
                                'searchFormCollapsed',
                                searchFormCollapsedClassName,
                            )}
                            between={2}
                            inline
                            nowrap
                        >
                            <TrainsSearchInformation
                                onClick={onCollapsedSearchFormClick}
                                deviceType={deviceType}
                            />

                            {filters}
                        </Flex>
                        {fastFilters}
                    </Flex>
                );
            case EProjectName.HOTELS: {
                return (
                    <HotelsSearchInformation
                        className={cx(
                            'searchFormCollapsed',
                            searchFormCollapsedClassName,
                        )}
                        onClick={onCollapsedSearchFormClick}
                        deviceType={deviceType}
                    />
                );
            }
            case EProjectName.BUSES: {
                return (
                    <Flex
                        className={cx(
                            'searchFormCollapsed',
                            searchFormCollapsedClassName,
                        )}
                        between={2}
                        inline
                        nowrap
                    >
                        <BusesSearchInformation
                            onClick={onCollapsedSearchFormClick}
                        />

                        {filters}
                    </Flex>
                );
            }

            default:
                return null;
        }
    }, [
        hideSearchInformation,
        project,
        aviaNewTouchFilters,
        searchFormCollapsedClassName,
        onCollapsedSearchFormClick,
        deviceType,
        filters,
        fastFilters,
    ]);

    const searchFormBlock = useMemo(() => {
        switch (project) {
            case EProjectName.AVIA:
                return (
                    <AviaSearchForm
                        className={cx('searchForm')}
                        onSubmit={handleSubmitForm}
                        useFilters
                        validateOnMount={validateSearchFormOnMount}
                        {...searchFormProps}
                    />
                );
            case EProjectName.TRAINS:
                return (
                    <TrainsSearchForm
                        className={cx('searchForm')}
                        onSubmit={handleSubmitForm}
                        {...searchFormProps}
                    />
                );
            case EProjectName.HOTELS: {
                return (
                    <HotelsSearchForm
                        className={cx('searchForm')}
                        onSubmit={handleSubmitForm}
                        {...searchFormProps}
                    />
                );
            }

            case EProjectName.BUSES:
                return (
                    <BusesSearchForm
                        className={cx('searchForm')}
                        onSubmit={handleSubmitForm}
                        {...searchFormProps}
                    />
                );

            default:
                return null;
        }
    }, [handleSubmitForm, project, validateSearchFormOnMount, searchFormProps]);

    const topBlock = useMemo(() => {
        if (hideTopMobileBlock) {
            return null;
        }

        return (
            <div className={cx('wrapper', 'wrapper_mobile')}>
                <div className={cx('main')}>
                    {isWhiteLabel ? (
                        <WhiteLabelLogo project={project} />
                    ) : (
                        <YandexTravelLogo
                            className={cx('logo')}
                            isSmall
                            project={project}
                            {...prepareQaAttributes(portalLogoQa)}
                        />
                    )}

                    <UserInfo
                        hasSideSheetNavigation={hasSideSheetNavigation}
                        withNavCaptions={withNavCaptions}
                        showFavoritesLink={showFavoritesLink}
                        {...prepareQaAttributes(userInfoQa)}
                    />

                    <NavigationSideSheet
                        className={cx('navigationSideSheet')}
                        {...prepareQaAttributes(navigationSideSheetQa)}
                    />
                </div>
            </div>
        );
    }, [
        hasSideSheetNavigation,
        hideTopMobileBlock,
        isWhiteLabel,
        navigationSideSheetQa,
        portalLogoQa,
        project,
        showFavoritesLink,
        userInfoQa,
        withNavCaptions,
    ]);

    return (
        <header
            className={cx(
                'root',
                className,
                `root_project_${project}`,
                `root_border_${borderBottomType}`,
            )}
            {...prepareQaAttributes(props)}
        >
            {topBlock}

            {showNavigation && !hasSideSheetNavigation && (
                <div className={cx('navigationWrapper')}>
                    <div className={cx('navigationScroller')}>
                        <NavigationContainer
                            className={cx('navigation')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'navigation',
                            })}
                        />
                    </div>
                </div>
            )}
            {searchFormIsAvailable &&
                project &&
                SUPPORTED_PROJECTS.includes(project) && (
                    <>
                        <div>
                            <div className={cx('wrapper', 'wrapper_mobile')}>
                                {searchInformationBlock}
                            </div>
                        </div>

                        <ModalWithHistoryBack
                            isMobile
                            fullScreen
                            disableAutoFocus
                            isVisible={modalIsVisible}
                            verticalAlign="top"
                            autoclosable={false}
                            animation={EModalAnimationType.NONE}
                            onClose={hideModal}
                        >
                            <Modal.Content>{searchFormBlock}</Modal.Content>
                        </ModalWithHistoryBack>
                    </>
                )}
        </header>
    );
};

export default HeaderMobile;
