import React, {useCallback} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {URLs} from 'constants/urls';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {IWithClassName} from 'types/withClassName';
import {EUserAvatarSize} from 'components/User/types';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';

import * as mailActions from 'reducers/common/userUnreadMailCount/actions';

import {getUserUndeadMailCount} from 'selectors/common/getUserUndeadMailCount';

import {internalUrl} from 'utilities/url';
import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useSingleProjectView} from 'utilities/hooks/useSingleProjectView';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useUniqSvg} from 'utilities/hooks/useUniqSvg';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {deviceMods} from 'utilities/stylesUtils';
import {useIsOnPage} from 'utilities/hooks/useIsOnPage';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';

import * as i18nBlock from 'i18n/index';

import User from 'components/User/User';
import Link from 'components/Link/Link';
import Flex from 'components/Flex/Flex';
import YandexMessengerLink from 'components/YandexMessenger/components/YandexMessengerLink/YandexMessengerLink';
import PlusHeaderIcon from './components/PlusHeaderIcon/PlusHeaderIcon';
import HeartFilledIcon from 'icons/24/HeartFilled';
import HeartOutlineIcon from 'icons/24/HeartOutline';
import Text from 'components/Text/Text';
import TripsLink from 'components/Header/components/UserInfo/components/TripsLink/TripsLink';
import CertificateLink from 'components/Header/components/UserInfo/components/CertificateLink/CertificateLink';

import cx from './UserInfo.scss';

const accountClickHandler = (): void => reachGoal(ECommonGoal.ACCOUNT_TAB);

interface IUserInfoProps extends IWithClassName, IWithQaAttributes {
    hasSideSheetNavigation?: boolean;
    withNavCaptions?: boolean;
    showFavoritesLink?: boolean;
}

const UserInfo: React.FC<IUserInfoProps> = props => {
    const {
        className,
        hasSideSheetNavigation,
        withNavCaptions,
        showFavoritesLink = true,
    } = props;

    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const userInfo = useUserInfo();
    const isSingleProjectView = useSingleProjectView();
    const unreadMailCount = useSelector(getUserUndeadMailCount);
    const fetchUserUnreadMailCount = useCallback(() => {
        dispatch(mailActions.fetchUserUnreadMailCount());
    }, [dispatch]);
    const ref = useUniqSvg<HTMLDivElement>();
    const {isWhiteLabel} = useWhiteLabelConfig();

    const {isMobile, isDesktop} = deviceType;

    const showTickets =
        isDesktop && !hasSideSheetNavigation && !isSingleProjectView;
    const isAuth = isAuthUser(userInfo);

    const showUser = isDesktop && !(hasSideSheetNavigation && isAuth);

    const isOnAccountPage = useIsOnPage(URLs.account, {exact: false});
    const isOnFavoritesPage = useIsOnPage(URLs.favorites);
    const isOnHotelsHappyPage = useIsOnPage(URLs.hotelsBookSuccess);
    const isOnTransportHappyPage = useIsOnPage([
        URLs.aviaBookingResult,
        URLs.trainsOrderHappyPage,
        URLs.busesBookHappyPage,
    ]);
    const isOnHappyPage = isOnHotelsHappyPage || isOnTransportHappyPage;

    const onFavoritesClick = useCallback(() => {
        reachGoal(EFavoritesGoal.HEADER_FAVORITES_CLICK);
    }, []);

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), className)}
            ref={ref}
            {...prepareQaAttributes(props)}
        >
            <Flex
                className={cx('icons')}
                between={isMobile || !withNavCaptions ? 4 : 5}
                inline
                alignItems="center"
            >
                {isMobile && !isWhiteLabel && (
                    <PlusHeaderIcon
                        className={cx('plus')}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'plus',
                        })}
                    />
                )}

                {isDesktop && !isWhiteLabel && (
                    <CertificateLink
                        title={i18nBlock.navCertificates()}
                        label={
                            withNavCaptions
                                ? i18nBlock.navCertificates()
                                : undefined
                        }
                    />
                )}

                {isDesktop && (
                    <YandexMessengerLink
                        entrypoint="header"
                        metrikaGoal={ECommonGoal.PORTAL_CHAT_HEADER_CLICK}
                        title={i18nBlock.navSupport()}
                        label={
                            withNavCaptions ? i18nBlock.navSupport() : undefined
                        }
                        labelSize="s"
                    />
                )}

                {showFavoritesLink && (
                    <Link
                        to={internalUrl(URLs.favorites)}
                        theme="black"
                        title={i18nBlock.navFavorites()}
                        onClick={onFavoritesClick}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'favoriteLink',
                        })}
                    >
                        <Flex inline alignItems="center" between={2}>
                            {isOnFavoritesPage ? (
                                <HeartFilledIcon />
                            ) : (
                                <HeartOutlineIcon />
                            )}

                            {withNavCaptions && (
                                <Text
                                    className={cx('text')}
                                    color="current"
                                    size="s"
                                >
                                    {i18nBlock.navFavorites()}
                                </Text>
                            )}
                        </Flex>
                    </Link>
                )}

                {showTickets && (
                    <TripsLink
                        to={internalUrl(URLs.accountTrips)}
                        title={i18nBlock.navTrips()}
                        isOnHappyPage={isOnHappyPage}
                        isOnHotelsHappyPage={isOnHotelsHappyPage}
                        isActive={isOnAccountPage}
                        onClick={accountClickHandler}
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'accountLink',
                        })}
                    >
                        {withNavCaptions ? i18nBlock.navTrips() : undefined}
                    </TripsLink>
                )}
            </Flex>

            {isDesktop && !isWhiteLabel && (
                <PlusHeaderIcon
                    className={cx('plus')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'plus',
                    })}
                />
            )}

            {showUser && (
                <div
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'login',
                    })}
                >
                    <User
                        className={cx('user')}
                        userInfo={userInfo}
                        avatarSize={
                            isMobile ? EUserAvatarSize.S : EUserAvatarSize.L
                        }
                        unreadMailCount={unreadMailCount}
                        fetchUserUnreadMailCount={fetchUserUnreadMailCount}
                    />
                </div>
            )}
        </div>
    );
};

export default UserInfo;
