import {FC, MouseEventHandler, memo, useEffect, useRef, ReactNode} from 'react';
import {useSelector} from 'react-redux';

import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {isNewTripSelector} from 'selectors/trips/tripPage/isNewTripSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useIsAuth} from 'utilities/hooks/useIsAuth';

import * as i18nBlock from 'i18n/trips-common';

import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import MessageBoxPopup, {
    EMessageBoxPopupTheme,
} from 'components/MessageBoxPopup/MessageBoxPopup';
import NotificationDot from 'components/NotificationDot/NotificationDot';
import OrdersFilledIcon from 'icons/24/OrdersFilled';
import OrdersIcon from 'icons/24/Orders';

import cx from './TripsLink.scss';

interface ITripsLinkProps extends IWithQaAttributes {
    to: string;
    title: string;
    isOnHappyPage: boolean;
    isOnHotelsHappyPage: boolean;
    isActive: boolean;
    children: ReactNode;
    onClick?: MouseEventHandler;
}

const POPUP_DIRECTIONS = [
    EPopupDirection.BOTTOM_RIGHT,
    EPopupDirection.BOTTOM,
    EPopupDirection.BOTTOM_LEFT,
];

const TripsLink: FC<ITripsLinkProps> = props => {
    const {
        to,
        title,
        isOnHappyPage,
        isOnHotelsHappyPage,
        isActive,
        children,
        onClick,
    } = props;

    const {
        value: isTooltipVisible,
        setTrue: openTooltip,
        setFalse: closeTooltip,
    } = useBoolean(false);
    const anchorRef = useRef<HTMLDivElement | null>(null);
    const {isDesktop} = useDeviceType();
    const isNewTrip = useSelector(isNewTripSelector);
    const isAuth = useIsAuth();

    useEffect(() => {
        if (isOnHappyPage && isNewTrip) {
            openTooltip();
        }
    }, [isNewTrip, isOnHappyPage, openTooltip]);

    return (
        <>
            <Link
                to={to}
                theme="black"
                onClick={onClick}
                title={title}
                {...prepareQaAttributes(props)}
            >
                <Flex
                    inline
                    alignItems="center"
                    between={2}
                    innerRef={anchorRef}
                >
                    <NotificationDot
                        className={cx('notificationDot')}
                        size="m"
                        dotVisible={isAuth && isNewTrip}
                    >
                        {isActive ? <OrdersFilledIcon /> : <OrdersIcon />}{' '}
                    </NotificationDot>
                    {Boolean(children) && (
                        <Text className={cx('text')} color="current" size="s">
                            {children}
                        </Text>
                    )}
                </Flex>
            </Link>

            {isDesktop && isAuth && (
                <MessageBoxPopup
                    theme={EMessageBoxPopupTheme.WHITE}
                    isVisible={isTooltipVisible}
                    anchorRef={anchorRef}
                    direction={POPUP_DIRECTIONS}
                    secondaryOffset={20}
                    onClose={closeTooltip}
                >
                    <div className={cx('tooltipContent')}>
                        <Text size="m">
                            {isOnHotelsHappyPage
                                ? i18nBlock.newTripHotelDescription()
                                : i18nBlock.newTripTicketDescription()}
                        </Text>
                    </div>
                </MessageBoxPopup>
            )}
        </>
    );
};

export default memo(TripsLink);
