import React from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Text, {ITextProps, TTextWeight, TTextSize} from 'components/Text/Text';

import cx from './Heading.scss';

export type THeadingLevel = '1' | '2' | '3' | 1 | 2 | 3;

interface IHeadingProps
    extends IWithClassName,
        Pick<ITextProps, 'tagProps' | 'size' | 'weight' | 'onClick'>,
        IWithQaAttributes {
    level: THeadingLevel;
}

/** Компонент заголовков h1-h3 на портале со стилизацией */
const Heading: React.FC<IHeadingProps> = ({
    className,
    weight,
    level,
    size,
    tagProps,
    children,
    ...rest
}) => {
    const levelTag = getHeadingTag(level);
    const levelSize = getLevelSize(level);
    const levelWeight = getLevelWeight(level, weight);

    return (
        <Text
            {...tagProps}
            {...rest}
            {...prepareQaAttributes(rest)}
            className={cx('root', className)}
            tag={levelTag}
            size={size || levelSize}
            weight={levelWeight}
        >
            {children}
        </Text>
    );
};

function getHeadingTag(level: THeadingLevel): 'h1' | 'h2' | 'h3' {
    switch (level) {
        case '1':
        case 1:
            return 'h1';
        case '2':
        case 2:
            return 'h2';
        case '3':
        case 3:
            return 'h3';
    }
}

function getLevelWeight(
    level: THeadingLevel,
    weight?: TTextWeight,
): TTextWeight {
    /** По гайду размер шрифта мы задаем только третьему уровню */
    if (level === 3) {
        if (weight) {
            return weight;
        }

        return 'medium';
    }

    return 'bold';
}

function getLevelSize(level: THeadingLevel): TTextSize {
    switch (level) {
        case '1':
        case 1:
            return 'xxl';
        case '2':
        case 2:
            return 'xl';
        case '3':
        case 3:
            return 'l';
    }
}

export default Heading;
