import React, {createRef, useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useEventListener} from 'utilities/hooks/useEventListener';

import cx from './HorizontalHiddenScroll.scss';

const cancelEventOptions: AddEventListenerOptions = {
    passive: true,
};

interface IHorizontalHiddenScrollProps
    extends IWithClassName,
        IWithQaAttributes,
        React.HTMLAttributes<HTMLDivElement> {
    scrollableClassName?: string;
    scrollerRef?: React.RefObject<HTMLDivElement>;
    /**
     * Запрещает передачу события touchmove выше.
     * @default false
     */
    stopPropagationTouchmove?: boolean;
    onScroll?: React.UIEventHandler<HTMLDivElement>;
}

const HorizontalHiddenScroll: React.FC<IHorizontalHiddenScrollProps> =
    props => {
        const {
            className,
            children,
            scrollableClassName,
            scrollerRef,
            stopPropagationTouchmove = false,
            onScroll,
            ...rest
        } = props;
        const contentRef = createRef<HTMLDivElement>();

        const cancelEvent = useCallback(
            event => {
                if (!stopPropagationTouchmove) {
                    return;
                }

                event.stopPropagation();
            },
            [stopPropagationTouchmove],
        );

        useEventListener(
            'touchmove',
            cancelEvent,
            contentRef,
            cancelEventOptions,
        );

        return (
            <div
                className={cx('root', className)}
                {...rest}
                {...prepareQaAttributes(props)}
                ref={contentRef}
            >
                <div
                    className={cx('scrollable', scrollableClassName)}
                    onScroll={onScroll}
                    ref={scrollerRef}
                >
                    {children}
                </div>
            </div>
        );
    };

export default HorizontalHiddenScroll;
