import React, {PureComponent} from 'react';
import {IYMapsApi, IYMapsInstance} from 'react-yandex-maps';
import {isEqual} from 'lodash';

/* Types */
import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {MapBoundsType} from 'types/common/ICoordinates';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';
import {IHotelWithMinPrice} from 'types/hotels/hotel/IHotelWithMinPrice';
import {IWithClassName} from 'types/withClassName';
import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

/* Components */
import YandexReadonlyMaps from 'components/YandexReadonlyMaps/YandexReadonlyMaps';
import MapHotelCard from 'projects/hotels/pages/SearchPage/components/HotelsSearchMap/components/MapHotelCard/MapHotelCard';
import HotelsMapStaticMarker from './components/HotelsMapStaticMarker';

/* Init styles */
import cx from './HotelsSearchStaticMap.scss';

interface IHotelsSearchStaticMapProps
    extends IWithQaAttributes,
        IWithClassName {
    mapClassName?: string;
    hotels: (IHotelWithOffers | IHotelWithMinPrice)[];
    bboxAsStruct: MapBoundsType;
    isMobile?: boolean;
    mapHeight?: string | number;
    mapWidth?: string | number;
    offerRequestParams?: IRequiredOfferParams;
    /* Markers */
    selectedMapMarkerPermalink?: PermalinkType;
    activeHotelPermalink?: PermalinkType;
    viewedMapMarkerPermalinks?: Record<PermalinkType, boolean>;
}

interface IHotelsSearchMapState {
    mapSuccessLoaded: boolean;
    canStartSearchAfterBoundsChange: boolean;
    activeMarkerPosition: Nullable<IActiveMarkerPosition>;
    bounds?: MapBoundsType;
    containerSize: {
        width: number;
        height: number;
    };
}

interface IActiveMarkerPosition {
    left: number;
    top: number;
    bottom: number;
}

class HotelsSearchStaticMap extends PureComponent<
    IHotelsSearchStaticMapProps,
    IHotelsSearchMapState
> {
    readonly state: IHotelsSearchMapState = {
        mapSuccessLoaded: false,
        activeMarkerPosition: null,
        canStartSearchAfterBoundsChange: true,
        containerSize: {
            width: 0,
            height: 0,
        },
    };

    private yandexMapAPIInstance: IYMapsApi | null = null;

    private yandexMapInstance: IYMapsInstance | null = null;

    /* Handlers */

    private handleLoadMap = (yandexMapAPIInstance: IYMapsApi): void => {
        if (yandexMapAPIInstance) {
            this.yandexMapAPIInstance = yandexMapAPIInstance;

            this.setState({mapSuccessLoaded: true});
            this.fitContainerToViewport();
            this.calculateContainerSize();
        }
    };

    private handleChangeActiveMarkerPosition = (
        markerPosition: IActiveMarkerPosition,
    ): void => {
        const {activeMarkerPosition} = this.state;

        if (
            !activeMarkerPosition ||
            !isEqual(activeMarkerPosition, markerPosition)
        ) {
            this.setState({
                activeMarkerPosition: markerPosition,
            });
        }
    };

    /* Helpers */

    private canRenderMarkerByMapBounds = (
        hotelWithMinPrice: IHotelWithOffers | IHotelWithMinPrice,
    ): boolean => {
        const {bounds} = this.state;
        const {
            hotel: {
                coordinates: {lat, lon},
            },
        } = hotelWithMinPrice;

        if (!bounds || !this.yandexMapAPIInstance) {
            return true;
        }

        const [leftBottomPoint, rightTopPoint] = bounds;

        /**
         * @link https://yandex.ru/dev/maps/jsapi/doc/2.1/ref/reference/util.bounds.html#method_detail__containsPoint
         */
        return this.yandexMapAPIInstance.util.bounds.containsPoint(
            [
                [leftBottomPoint.lat, leftBottomPoint.lon],
                [rightTopPoint.lat, rightTopPoint.lon],
            ],
            [lat, lon],
        );
    };

    /* Refs manipulate */

    private setYandexMapInstanceRef = (
        yandexMapInstance: IYMapsInstance,
    ): void => {
        this.yandexMapInstance = yandexMapInstance;
    };

    /* Actions */

    private fitContainerToViewport = (): void => {
        if (this.yandexMapInstance) {
            const yandexMapContainer = this.yandexMapInstance.container;

            yandexMapContainer.fitToViewport();
        }
    };

    private calculateContainerSize = (): void => {
        if (this.yandexMapInstance) {
            const yandexMapContainer = this.yandexMapInstance.container;
            const [width, height] = yandexMapContainer.getSize() || [0, 0];

            this.setState({
                containerSize: {
                    width,
                    height,
                },
            });
        }
    };

    /* Render */

    private renderHotelsMarkers(): React.ReactNode {
        const {mapSuccessLoaded} = this.state;
        const {hotels, activeHotelPermalink, selectedMapMarkerPermalink} =
            this.props;

        if (!mapSuccessLoaded) {
            return null;
        }

        return hotels.map(hotel => {
            if (!this.canRenderMarkerByMapBounds(hotel)) {
                return null;
            }

            return (
                <HotelsMapStaticMarker
                    key={hotel.hotel.permalink}
                    hotel={hotel}
                    isActiveMarker={
                        !selectedMapMarkerPermalink &&
                        activeHotelPermalink === hotel.hotel.permalink
                    }
                    yandexMapAPIInstance={this.yandexMapAPIInstance}
                    onSetMarkerPosition={this.handleChangeActiveMarkerPosition}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'marker',
                        key: hotel.hotel.permalink,
                    })}
                />
            );
        });
    }

    private renderActiveHotelCard(): React.ReactNode {
        const {hotels, activeHotelPermalink} = this.props;
        const {activeMarkerPosition, containerSize} = this.state;

        if (!activeMarkerPosition) {
            return null;
        }

        const activeHotel = hotels.find(
            hotel => hotel.hotel.permalink === activeHotelPermalink,
        );

        if (!activeHotel) {
            return null;
        }

        return (
            <MapHotelCard
                className={cx('hotelCard')}
                hotel={activeHotel}
                containerSize={containerSize}
                markerPosition={activeMarkerPosition}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'activeHotelCard',
                })}
            />
        );
    }

    render(): React.ReactNode {
        const {bboxAsStruct, className, mapClassName} = this.props;

        if (!bboxAsStruct) {
            return null;
        }

        return (
            <div
                className={cx('hotelsMap', className)}
                {...prepareQaAttributes(this.props)}
            >
                <YandexReadonlyMaps
                    className={mapClassName}
                    hideCopyright
                    defaultBounds={bboxAsStruct}
                    avoidFractionalZoom={false}
                    yandexMapDisablePoiInteractivity={true}
                    yandexMapInstanceRef={this.setYandexMapInstanceRef}
                    onLoadMap={this.handleLoadMap}
                    {...prepareQaAttributes(this.props)}
                >
                    {this.renderHotelsMarkers()}
                </YandexReadonlyMaps>

                {this.renderActiveHotelCard()}
            </div>
        );
    }
}

export default HotelsSearchStaticMap;
