import React, {ReactNode, useState, useRef, useEffect, useMemo} from 'react';
import classNames from 'classnames';

import {TResponsiveImage} from 'types/common/TImage';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useBoolean} from 'utilities/hooks/useBoolean';

import Carousel, {ICarouselProps} from 'components/Carousel/Carousel';
import CarouselImage from './components/CarouselImage/CarouselImage';

import cx from './ImagesCarousel.scss';

export interface IImageCarouselProps
    extends Omit<ICarouselProps, 'children' | 'lazyLoadSizes'> {
    images: TResponsiveImage[];

    height?: number;
    autoHeight?: boolean;

    imageClassName?: string;
    imageWrapClassName?: string;
    withoutImageClassName?: string;

    beforeNode?: ReactNode;
    isDesktop: boolean;
    /**
     * Грузить картинки при появлении их во вьюпорте
     */
    isLoadImageOnViewportIntersect?: boolean;

    onImageClick?: (src: string, index: number) => void;
    getItemWidth?: (index: number) => number | string | undefined;
}

/**
 * Компонент карусели для изображений.
 * По умолчанию высота карусели будет равна высоте наибольшего изображения.
 *
 * @comment Для lazyLoad необходимо передавать размеры картинок.
 * Также для lazyLoad происходит два рендера, первый для определения необходимого количества элементов. Второй встраивает картинки.
 * @bug Есть баг в FF при использовании нативного скролла в десктопе, некорректно расчитывается размер
 * контейнера https://bugzilla.mozilla.org/show_bug.cgi?id=1281713. Пока отложено, т.к. нативный скролл используем только на таче.
 */
const ImagesCarousel: React.FC<IImageCarouselProps> = props => {
    const {
        className,
        listClassName,
        itemClassName,
        imageClassName,
        imageWrapClassName,
        withoutImageClassName,
        isDesktop,
        type,
        getItemWidth,
        spaceBetween = 1,
        autoHeight = false,
        lazyLoad,
        height,
        images,
        beforeNode,
        isLoadImageOnViewportIntersect,
        onImageClick,
        ...carouselProps
    } = props;

    const {value: mounted, setTrue: setMounted} = useBoolean(false);

    const [containerHeight, setContainerHeight] = useState<number | undefined>(
        0,
    );

    const carouselRef = useRef<Carousel | null>(null);
    const containerRef = useRef<HTMLDivElement | null>(null);

    useEffect(() => {
        setMounted();

        setContainerHeight(
            autoHeight
                ? containerRef.current?.getBoundingClientRect().height
                : undefined,
        );
    }, [autoHeight, setMounted]);

    const placeholder = useMemo(() => {
        return (
            <CarouselImage
                className={cx('image', itemClassName)}
                imageClassName={imageClassName}
                withoutImageClassName={withoutImageClassName}
                containerHeight={containerHeight}
                isPlaceholder
            />
        );
    }, [containerHeight, imageClassName, itemClassName, withoutImageClassName]);

    const carouselContent = useMemo(() => {
        // Высота необходима из-за бага в ie11 и edge из-за которого ширина контейнера,
        // в котором находится изображение, будет равна оригинальной ширине изображения (много пустого места справа)
        const onlyOneImage = images.length === 1;
        const items: React.ReactNode[] = images.map(
            ({src, size, alt, additionalSizes}, index) => (
                <CarouselImage
                    src={src}
                    key={src}
                    alt={alt}
                    additionalSizes={additionalSizes}
                    index={index}
                    className={cx('image', itemClassName)}
                    imageClassName={imageClassName}
                    withoutImageClassName={withoutImageClassName}
                    full={type === 'mini' || onlyOneImage}
                    size={(lazyLoad && type !== 'mini' && size) || undefined}
                    containerHeight={height || containerHeight}
                    isLoadOnViewportIntersect={isLoadImageOnViewportIntersect}
                    onClick={onImageClick}
                    onLoad={carouselRef.current?.checkNextItems}
                    isRenderImmediately={index < 5}
                />
            ),
        );

        if (beforeNode) {
            items.unshift(beforeNode);
        }

        return items;
    }, [
        beforeNode,
        containerHeight,
        height,
        imageClassName,
        images,
        isLoadImageOnViewportIntersect,
        itemClassName,
        lazyLoad,
        onImageClick,
        type,
        withoutImageClassName,
    ]);

    const isCarouselVisible = !lazyLoad || autoHeight || mounted;

    return (
        <div
            className={classNames(cx('root'), className)}
            style={{
                height: height || (autoHeight && containerHeight) || undefined,
            }}
            ref={containerRef}
            {...prepareQaAttributes({
                current: 'imagesCarousel',
            })}
        >
            {isCarouselVisible && (
                <Carousel
                    ref={carouselRef}
                    scroll={isDesktop ? 'arrow' : 'native'}
                    type={type}
                    listClassName={listClassName}
                    itemClassName={cx(imageWrapClassName)}
                    lazyLoadSizes={images.map(x => x.size)}
                    lazyPlaceholder={placeholder}
                    getItemWidth={getItemWidth}
                    spaceBetween={spaceBetween}
                    lazyLoad={lazyLoad}
                    {...carouselProps}
                >
                    {carouselContent}
                </Carousel>
            )}
        </div>
    );
};

export default React.memo(ImagesCarousel);
