import {FunctionComponent, useCallback, useMemo} from 'react';
import classNames from 'classnames';

import {IWithClassName} from 'types/withClassName';
import {ISize, TImage} from 'types/common/TImage';

import {deviceModDesktop} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import TravelImage from 'components/TravelImage/TravelImage';

import cx from './CarouselImage.scss';

interface ICarouselImageCommonProps extends IWithClassName {
    /** Высота контейнера карусели для LazyLoad */
    containerHeight?: number;
    /** Размеры картинки для LazyLoad */
    size?: ISize;

    imageClassName?: string;
    withoutImageClassName?: string;
    full?: boolean;
    isLoadOnViewportIntersect?: boolean;
    onLoad?: () => void;
    isRenderImmediately?: boolean;
}

interface ICarouselImageSrcProps extends ICarouselImageCommonProps {
    src: string;
    alt?: string;
    additionalSizes?: TImage[];
    index: number;
    isPlaceholder?: never;
    onClick?: (src: string, index: number) => void;
}

interface ICarouselImagePlaceholderProps extends ICarouselImageCommonProps {
    src?: never;
    alt?: never;
    additionalSizes?: never;
    index?: never;
    isPlaceholder: true;
    onClick?: never;
}

export type TCarouselImageProps =
    | ICarouselImageSrcProps
    | ICarouselImagePlaceholderProps;

const CarouselImage: FunctionComponent<TCarouselImageProps> = ({
    src,
    alt,
    containerHeight,
    size,
    className,
    imageClassName,
    withoutImageClassName,
    full,
    index,
    isPlaceholder,
    isLoadOnViewportIntersect = false,
    onClick,
    onLoad,
    additionalSizes,
    isRenderImmediately,
}) => {
    const deviceType = useDeviceType();
    const style = useMemo(() => {
        if (containerHeight && size) {
            const aspectRatio = containerHeight / size.height;

            return {
                height: containerHeight,
                width: size.width * aspectRatio,
            };
        }
    }, [size, containerHeight]);

    const handleClick = useCallback(() => {
        if (onClick && src && typeof index === 'number') {
            onClick(src, index);
        }
    }, [onClick, src, index]);

    return (
        <TravelImage
            className={classNames(
                cx(
                    'root',
                    {root_with_hover: Boolean(onClick)},
                    {root_full: full},
                    deviceModDesktop('root', deviceType),
                ),
                className,
            )}
            imageClassName={cx('image', imageClassName)}
            withoutImageClassName={withoutImageClassName}
            src={src || ''}
            imageAlt={alt}
            responsiveSet={additionalSizes}
            style={style}
            imageStyle={{height: containerHeight}}
            isPlaceholder={isPlaceholder}
            isLoadOnViewportIntersect={isLoadOnViewportIntersect}
            onClick={handleClick}
            onLoad={onLoad}
            isRenderImmediately={isRenderImmediately}
            /*
             * Аттрибуты width, height нужны для SEO целей,
             * т.к. без этих аттрибутов них бот алертит что тут есть риск CLS
             */
            attrs={{width: style?.width, height: containerHeight}}
        />
    );
};

export default CarouselImage;
