import _ from 'lodash';

const THRESHOLD = 3;
const MAX_XS_DOTS = 1;

const getMaxSDots = (windowSize: number): number =>
    windowSize <= THRESHOLD ? 0 : 1;

const getLeftCornerBoundary = (windowSize: number): number =>
    windowSize - getMaxSDots(windowSize) - MAX_XS_DOTS - 1;

const getRightCornerBoundary = (
    windowSize: number,
    itemsCount: number,
): number => itemsCount - windowSize + MAX_XS_DOTS + getMaxSDots(windowSize);

export function getContainerOffset(
    itemsCount: number,
    windowSize: number,
    pointSize: number,
    activeItem: number,
): number {
    const leftCornerBoundary = getLeftCornerBoundary(windowSize);

    // Первые N точек перекрашиваются, но при этом контейнер не движется
    if (activeItem <= leftCornerBoundary) {
        return 0;
    }

    // В середине появляется смещение, которое останавливается на последних N точках
    const rightCornerBoundary = itemsCount - windowSize;
    const halfSize = Math.floor(windowSize / 2);

    return _.clamp(
        pointSize * (activeItem - halfSize),
        0,
        pointSize * rightCornerBoundary,
    );
}

export function getDotSize(
    items: number,
    // нечетное
    visibleItemsCount: number,
    activeItem: number,
    index: number,
): string | undefined {
    const maxSDots = getMaxSDots(visibleItemsCount);

    const leftCornerBoundary = getLeftCornerBoundary(visibleItemsCount);
    const isLeftCornerCase = activeItem <= leftCornerBoundary;

    // Кейс, когда активная точка среди первых N точек
    if (isLeftCornerCase) {
        const isExtraSmall =
            index >= leftCornerBoundary + MAX_XS_DOTS + maxSDots;

        if (isExtraSmall) return 'xs';

        const isSmall = index >= leftCornerBoundary + MAX_XS_DOTS;

        if (isSmall) return 's';

        return;
    }

    const rightCornerBoundary = getRightCornerBoundary(
        visibleItemsCount,
        items,
    );
    const isRightCornerCase = activeItem >= rightCornerBoundary;

    // Кейс, когда активная точка среди последних N точек
    if (isRightCornerCase) {
        const isExtraSmall =
            index <= rightCornerBoundary - MAX_XS_DOTS - maxSDots;

        if (isExtraSmall) return 'xs';

        const isSmall = index <= rightCornerBoundary - MAX_XS_DOTS;

        if (isSmall) return 's';

        return;
    }

    // Активная точка где-то в середине
    const halfSize = Math.floor(visibleItemsCount / 2);
    const isDotInMiddle = _.inRange(
        index,
        activeItem - halfSize + MAX_XS_DOTS + maxSDots,
        activeItem + halfSize - maxSDots - MAX_XS_DOTS + 1,
    );

    if (isDotInMiddle) return;

    const isSmall = _.inRange(
        index,
        activeItem - halfSize + maxSDots,
        activeItem + halfSize - maxSDots + 1,
    );

    if (isSmall && maxSDots) return 's';

    return 'xs';
}
