import {FunctionComponent, ReactNode, useCallback, Fragment} from 'react';
import {Slider, Handles, Rail, Tracks} from 'react-compound-slider';

import {IWithClassName} from 'src/types/withClassName';

/* Components */
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import InputRangeHandle from './components/InputRangeHandle/InputRangeHandle';
import InputRangeTrack from './components/InputRangeTrack/InputRangeTrack';
import InputRangeRail from './components/InputRangeRail/InputRangeRail';
import InputRangeTooltip from './components/InputRangeTooltip/InputRangeTooltip';

/* Init styles */
import cx from './InputRange.scss';

/* Component Types */
type TValue = number;
type TValues = readonly TValue[];
type TRangeValues = [TValue, TValue];
type TTooltipAlignment = 'left' | 'right';

export interface IInputRangeProps extends IWithClassName, IWithQaAttributes {
    /**
     * Граничные значения
     */
    rangeValues: TRangeValues;
    /**
     * Выбранные значения
     */
    values: TValues;
    /**
     * Ignore all mouse, touch and keyboard events.
     */
    disabled?: boolean;
    /**
     * Шаг ползунка
     *
     * @default 1
     */
    step?: number;
    /**
     * Мод для работы react-compound-slider
     *
     * @default 2
     */
    mode?: number;
    /**
     * Размер InputRangeHandle
     *
     * @default 's'
     */
    size?: 'm' | 's';
    /**
     * Рисовать ли слева от ползука Track
     *
     * @default false
     */
    hasLeftTrack?: boolean;
    /**
     * Рисовать ли справа от ползука Track
     *
     * @default false
     */
    hasRightTrack?: boolean;
    /**
     * Позиция тултипа относительно Rail и Handle
     *
     * @default 'top'
     */
    tooltipPosition?: 'top' | 'bottom';
    /**
     * Тема тултипа
     *
     * @default 'clear'
     */
    tooltipTheme?: 'white' | 'clear';
    /**
     * Рендерить ли тултип
     *
     * @default false
     */
    canRenderTooltip?: boolean;
    /**
     * Фунция для рендеринга контента тултипа
     */
    disableKeyboard?: boolean;
    renderHandleTooltip?: (renderProps: {
        value: TValue;
        rangeValues: TRangeValues;
        alignment: TTooltipAlignment;
    }) => ReactNode;
    /**
     * Коллбек при окончании изменений
     */
    onChange?: (changedValues: TValues) => void;
    /**
     * Коллбек в процессе изменений
     */
    onUpdate?: (updatedValues: TValues) => void;
}

/* Rail Component */

/* Input Range Component */
const InputRange: FunctionComponent<IInputRangeProps> = props => {
    const {
        className,
        step,
        mode,
        size,
        values,
        disabled,
        onChange,
        onUpdate,
        rangeValues,
        canRenderTooltip,
        tooltipTheme,
        hasLeftTrack,
        hasRightTrack,
        tooltipPosition,
        renderHandleTooltip,
        disableKeyboard = false,
    } = props;

    const handleChange = useCallback(
        (changedValues: TValues): void => {
            if (onChange) {
                onChange(changedValues);
            }
        },
        [onChange],
    );

    const handleUpdate = useCallback(
        (updatedValues: TValues): void => {
            if (onUpdate) {
                onUpdate(updatedValues);
            }
        },
        [onUpdate],
    );

    return (
        <Slider
            className={cx('range', `range_size_${size}`, className)}
            domain={rangeValues}
            values={values}
            onChange={handleChange}
            onUpdate={handleUpdate}
            mode={mode}
            step={step}
            disabled={disabled}
        >
            <Rail>
                {(railProps): ReactNode => (
                    <InputRangeRail
                        size={size}
                        disabled={disabled}
                        railProps={railProps}
                    />
                )}
            </Rail>
            <Handles>
                {({handles, getHandleProps, activeHandleID}): ReactNode => (
                    <>
                        {handles.map((handle, index) => (
                            <Fragment key={index}>
                                <InputRangeHandle
                                    key={handle.id}
                                    handle={handle}
                                    size={size}
                                    isFirst={index === 0}
                                    activeHandleID={activeHandleID}
                                    disabled={disabled}
                                    getHandleProps={getHandleProps}
                                    disableKeyboard={disableKeyboard}
                                    {...prepareQaAttributes({
                                        parent: props,
                                        current: `handle`,
                                        key: index,
                                    })}
                                />
                                {canRenderTooltip && renderHandleTooltip && (
                                    <InputRangeTooltip
                                        handle={handle}
                                        rangeValues={rangeValues}
                                        isFirst={index === 0}
                                        size={size}
                                        tooltipTheme={tooltipTheme}
                                        getHandleProps={getHandleProps}
                                        activeHandleID={activeHandleID}
                                        tooltipPosition={tooltipPosition}
                                        renderHandleTooltip={
                                            renderHandleTooltip
                                        }
                                    />
                                )}
                            </Fragment>
                        ))}
                    </>
                )}
            </Handles>
            <Tracks left={hasLeftTrack} right={hasRightTrack}>
                {({tracks, getTrackProps}): ReactNode => (
                    <>
                        {tracks.map(({id, source, target}) => (
                            <InputRangeTrack
                                key={id}
                                id={id}
                                size={size}
                                source={source}
                                target={target}
                                getTrackProps={getTrackProps}
                                disabled={disabled}
                            />
                        ))}
                    </>
                )}
            </Tracks>
        </Slider>
    );
};

InputRange.defaultProps = {
    step: 1,
    mode: 2,
    size: 's',
    hasLeftTrack: false,
    tooltipTheme: 'clear',
    canRenderTooltip: true,
    tooltipPosition: 'top',
    hasRightTrack: false,
};

export default InputRange;
