import {FunctionComponent, useEffect, useRef} from 'react';
import {SliderItem, GetHandleProps} from 'react-compound-slider';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import {IInputRangeProps} from 'components/InputRange/InputRange';

/* Init Styles */
import cx from './InputRangeHandle.scss';

/* Constants */
const PERCENT_FOR_FIRST_ACTIVE_MODE = 70;

/* Component Types */
interface IInputRangeHandleProps extends IWithQaAttributes {
    handle: SliderItem;
    getHandleProps: GetHandleProps;
    activeHandleID: string;
    disabled?: boolean;
    isFirst: boolean;
    size: IInputRangeProps['size'];
    disableKeyboard?: boolean;
}

const InputRangeTooltip: FunctionComponent<IInputRangeHandleProps> = ({
    handle: {id, value, percent},
    size,
    isFirst,
    getHandleProps,
    activeHandleID,
    disabled,
    disableKeyboard,
    ...rest
}) => {
    const lastActiveHandleIDRef = useRef('');

    useEffect(() => {
        if (activeHandleID) {
            lastActiveHandleIDRef.current = activeHandleID;
        }
    }, [activeHandleID, lastActiveHandleIDRef]);

    const {current: lastActiveHandleID} = lastActiveHandleIDRef;
    const canUseLastActiveAfterMount =
        isFirst && percent > PERCENT_FOR_FIRST_ACTIVE_MODE;

    return (
        <button
            className={cx('handle', `handle_size_${size}`, {
                handle_active: activeHandleID === id,
                handle_lastActive: lastActiveHandleID
                    ? lastActiveHandleID === id
                    : canUseLastActiveAfterMount,
            })}
            style={{
                left: `${percent}%`,
            }}
            type="button"
            aria-valuenow={value}
            disabled={disabled}
            {...getHandleProps(id)}
            tabIndex={disableKeyboard ? -1 : undefined}
            {...prepareQaAttributes(rest)}
        >
            <div className={cx('handleCircle')} />
        </button>
    );
};

export default InputRangeTooltip;
