import React, {useCallback} from 'react';

import {IInputProps} from 'components/Input/types/InputProps';

import copyToClipboard from 'utilities/copyToClipboard';

import * as i18n from 'i18n/common-share';

import Input from 'components/Input/Input';
import Button from 'components/Button/Button';
import Flex from 'components/Flex/Flex';
import CopyIcon from 'icons/16/Copy';
import CheckCircleIcon from 'icons/16/CheckCircle';

import cx from './InputWithCopyButton.scss';

interface IInputWithCopyButtonProps extends IInputProps {
    isCopied: boolean;
    onIsCopiedChange(isCopied: boolean): void;
}

const InputWithCopyButton: React.FC<IInputWithCopyButtonProps> = props => {
    const {className, isCopied, onIsCopiedChange, value, onChange, ...rest} =
        props;

    const handleChangeInput: IInputProps['onChange'] = useCallback(
        (e, v) => {
            onChange?.(e, v);

            onIsCopiedChange(false);
        },
        [onChange, onIsCopiedChange],
    );

    const handleButtonClick = useCallback(async () => {
        if (!value) {
            return;
        }

        const copySuccess = await copyToClipboard({text: String(value)});

        onIsCopiedChange(copySuccess);
    }, [onIsCopiedChange, value]);

    return (
        <Flex className={className} inline between={1} nowrap>
            <Input
                value={value}
                onChange={handleChangeInput}
                iconRight={
                    isCopied ? (
                        <CheckCircleIcon className={cx('copiedIcon')} />
                    ) : undefined
                }
                {...rest}
            />
            <Button
                disabled={!value}
                iconLeft={<CopyIcon />}
                onClick={handleButtonClick}
            >
                {i18n.copy()}
            </Button>
        </Flex>
    );
};

export default React.memo(InputWithCopyButton);
