import React, {FC, useMemo} from 'react';
import _noop from 'lodash/noop';

import {IWithClassName} from 'types/withClassName';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useToggle} from 'utilities/hooks/useToggle';
import {deviceMods} from 'utilities/stylesUtils';

import CloseCircleIcon from 'icons/16/CloseCircle';

import cx from './Intent.scss';

export enum EIntentTheme {
    PRIMARY = 'primary',
    ALERT = 'alert',
}

/**
 * Комментарий (вторую строку в интенте) можно добавить
 * только для размеров m-inset и l
 */
interface IIntentWithoutCommentProps {
    comment?: never;
    /**
     * Размеры, которые доступны
     * @default m
     */
    size?: 's' | 'm' | 'm-inset' | 'l' | 'xl';
}

interface IIntentWithCommentProps {
    comment: string;
    size: 'm-inset' | 'l';
}

type TIntentCommentType = IIntentWithCommentProps | IIntentWithoutCommentProps;

/**
 * Есть либо свойство onClick либо onCancel либо оба
 * Одно из них должно быть обязательно
 */
interface IIntentClickNoCancleProps {
    onClick: React.MouseEventHandler<HTMLButtonElement>;
    onCancel?: never;
}

interface IIntentCancelNoClickProps {
    onClick?: never;
    onCancel: React.MouseEventHandler<HTMLButtonElement>;
}

interface IIntentCancelAndClickProps {
    onClick: React.MouseEventHandler<HTMLButtonElement>;
    onCancel: React.MouseEventHandler<HTMLButtonElement>;
}

type TIntentActionType =
    | IIntentClickNoCancleProps
    | IIntentCancelNoClickProps
    | IIntentCancelAndClickProps;

/**
 * Можно вставить иконку, но в этом случае нельзя добавить текст
 * Текст и комментарий можно добавить только без иконки
 */
export interface IIntentWithText {
    text: React.ReactNode;
    comment?: string;
    icon?: never;
}

interface IIntentWithIcon {
    text?: never;
    comment?: never;
    icon: React.ReactNode;
    onCancel?: never;
}

type TIntentContentType = IIntentWithText | IIntentWithIcon;

type TIntentProps = TIntentCommentType &
    TIntentActionType &
    TIntentContentType &
    IWithClassName &
    IWithQaAttributes & {
        /**
         * Дизейбл кнопки
         */
        isDisabled?: boolean;
        /**
         * Можно выделить кнопку (находил применение только в интентах Ж/Д)
         */
        isChecked?: boolean;

        /**
         * Ширина элемента
         * auto - по содержимому
         * max - занимает все возможное место внутри родителя
         * @default auto
         */
        width?: 'auto' | 'max';
        theme?: EIntentTheme;
    };

export const Intent: FC<TIntentProps> = props => {
    const {
        size = 'm',
        width = 'auto',
        theme = 'primary',
        className,
        isDisabled = false,
        isChecked = false,
        onClick = _noop,
        icon,
        text,
        comment,
        onCancel,
    } = props;

    const deviceType = useDeviceType();
    const [cancelFocused, toggleCancelFocused] = useToggle(false);

    const currentSize: string = useMemo(() => {
        if (comment) {
            if (size === 'm-inset') {
                return 'm-with-comment';
            } else if (size === 'l') {
                return 'l-with-comment';
            }
        }

        return size;
    }, [size, comment]);

    const renderIcon = useMemo(
        () => <div className={cx('icon')}>{icon}</div>,
        [icon],
    );

    const renderText = useMemo(
        () => (
            <>
                <span className={cx('text')}>
                    <span className={cx('label')}>{text}</span>
                    {comment && (
                        <span className={cx('comment')}>{comment}</span>
                    )}
                </span>
                {onCancel && (
                    <button
                        className={cx('cancel')}
                        type="button"
                        onClick={onCancel}
                        onFocus={toggleCancelFocused}
                        onBlur={toggleCancelFocused}
                    >
                        <CloseCircleIcon />
                    </button>
                )}
            </>
        ),
        [text, comment, onCancel, toggleCancelFocused],
    );

    const content = (
        <span className={cx('content')}>{icon ? renderIcon : renderText}</span>
    );
    const classNames = cx(
        'intent',
        cancelFocused && 'intent_focused',
        deviceMods('intent', deviceType),
        `intent_size_${currentSize}`,
        `intent_width_${width}`,
        `intent_theme_${theme}`,
        className,
        {
            intent_withCancel: Boolean(onCancel),
            intent_allClickable: Boolean(onCancel && onClick),
            intent_checked: isChecked,
        },
    );

    if (onCancel) {
        return (
            <span className={classNames} {...prepareQaAttributes(props)}>
                {content}
            </span>
        );
    }

    return (
        <button
            className={classNames}
            type="button"
            disabled={isDisabled}
            onClick={onClick}
            {...prepareQaAttributes(props)}
        >
            {content}
        </button>
    );
};

export default Intent;
