import {FC, memo, useMemo} from 'react';

import {URLs} from 'constants/urls';

import {IWithClassName} from 'types/withClassName';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {useIsAuth} from 'utilities/hooks/useIsAuth';

import * as i18nBlock from 'i18n/happyPage';

import Flex from 'components/Flex/Flex';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import OrdersIcon from 'icons/16/Orders';

import cx from './SuccessText.scss';

interface ISuccessTextProps extends IWithClassName, IWithQaAttributes {
    title?: string;
    email: string;
}

const SuccessText: FC<ISuccessTextProps> = props => {
    const {
        className,
        title = i18nBlock.successTextTitleTicket(),
        email,
    } = props;

    const deviceType = useDeviceType();
    const isAuth = useIsAuth();

    const successText = useMemo(() => {
        if (!isAuth) {
            return i18nBlock.successTextDescriptionNotAuth({email});
        }

        return insertJSXIntoKey(i18nBlock.successTextDescriptionAuth)({
            email,
            tripsLink: (
                <Link className={cx('tripsLink')} to={URLs.accountTrips}>
                    <TextWithIcon
                        text={i18nBlock.trips()}
                        iconLeft={OrdersIcon}
                        spaceBetween={1}
                        inline
                    />
                </Link>
            ),
        });
    }, [email, isAuth]);

    return (
        <Flex
            className={cx('root', deviceMods('root', deviceType), className)}
            flexDirection="column"
            alignItems="flex-start"
            between={2}
            {...prepareQaAttributes(props)}
        >
            <Heading
                className={cx('title')}
                level={1}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'title',
                })}
            >
                {title}
            </Heading>

            <Text
                className={cx('description')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'description',
                })}
            >
                {successText}
            </Text>
        </Flex>
    );
};

export default memo(SuccessText);
