import React from 'react';
import {useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {IWithClassName} from 'types/withClassName';
import {EFooterProject} from 'components/Footer/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';

import {tabBarEnabledSelector} from 'selectors/common/tabBarEnabledSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Header from 'components/Header/Header';
import Footer from 'components/Footer/Footer';
import TabBar from 'components/TabBar/TabBar';
import {IBaseSearchFormProps} from 'components/SearchForm/SearchForm';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import cx from './LayoutDefault.scss';

export enum ELayoutBgColor {
    BASIC = 'basic',
    LIGHT = 'light',
    DARK = 'dark',
}

export interface ILayoutDefaultProps extends IWithClassName, IWithQaAttributes {
    headerClassName?: string;
    headerWrapperClassName?: string;
    headerBorderBottomType?: EHeaderBorderBottomType;
    formWrapperClassName?: string;
    footerWithoutSeparator?: boolean;
    contentClassName?: string;
    footerClassName?: string;
    footerWrapperClassName?: string;
    bgColor?: ELayoutBgColor;
    searchFormCollapsedClassName?: string;
    project?: EProjectName;
    footerType?: EFooterProject;
    showHeader?: boolean;
    showFooter?: boolean;
    showSearchForm?: boolean;
    showNavigation?: boolean;
    showTabBar?: boolean;
    isFixedNavigation?: boolean;
    hasSideSheetNavigation?: boolean;
    searchFormIsStatic?: boolean;
    initiallyCalendarIsOpen?: boolean;
    validateSearchFormOnMount?: boolean;
    searchFormInitialIsExpanded?: boolean;
    children: React.ReactNode;
    filters?: React.ReactNode;
    fastFilters?: React.ReactNode;
    hideTopMobileBlock?: boolean;
    hideSearchInformation?: boolean;
    footerIncludeSupportBlock?: boolean;
    isIndex?: boolean;
    onSearchFormSubmit?(): void;
    searchFormProps?: IBaseSearchFormProps;
}

const LayoutDefault: React.FC<ILayoutDefaultProps> = props => {
    const {
        className,
        headerClassName,
        headerWrapperClassName,
        headerBorderBottomType,
        formWrapperClassName,
        contentClassName,
        footerClassName,
        footerWrapperClassName,
        footerWithoutSeparator = false,
        bgColor = ELayoutBgColor.LIGHT,
        searchFormCollapsedClassName,
        project,
        footerType,
        showHeader = true,
        showFooter = true,
        showSearchForm,
        showNavigation,
        showTabBar,
        isFixedNavigation,
        hasSideSheetNavigation,
        searchFormIsStatic,
        initiallyCalendarIsOpen,
        validateSearchFormOnMount,
        searchFormInitialIsExpanded,
        children,
        filters,
        fastFilters,
        hideTopMobileBlock,
        hideSearchInformation,
        footerIncludeSupportBlock,
        isIndex,
        onSearchFormSubmit,
        searchFormProps,
    } = props;

    const tabBarEnabled = useSelector(tabBarEnabledSelector);

    return (
        <DeviceTypeContext.Consumer>
            {(deviceType): React.ReactNode => (
                <div
                    className={cx(
                        'root',
                        {
                            root_mobile: deviceType.isMobile,
                        },
                        `root_bgColor_${bgColor}`,
                        className,
                    )}
                    {...prepareQaAttributes(props)}
                >
                    {showHeader && (
                        <Header
                            className={headerClassName}
                            wrapperClassName={headerWrapperClassName}
                            formWrapperClassName={formWrapperClassName}
                            searchFormCollapsedClassName={
                                searchFormCollapsedClassName
                            }
                            isFixed={isFixedNavigation}
                            showSearchForm={showSearchForm}
                            showNavigation={showNavigation}
                            hasSideSheetNavigation={hasSideSheetNavigation}
                            searchFormIsStatic={searchFormIsStatic}
                            initiallyCalendarIsOpen={initiallyCalendarIsOpen}
                            validateSearchFormOnMount={
                                validateSearchFormOnMount
                            }
                            searchFormInitialIsExpanded={
                                searchFormInitialIsExpanded
                            }
                            onSubmitForm={onSearchFormSubmit}
                            project={project}
                            filters={filters}
                            fastFilters={fastFilters}
                            hideTopMobileBlock={hideTopMobileBlock}
                            hideSearchInformation={Boolean(
                                hideSearchInformation,
                            )}
                            borderBottomType={headerBorderBottomType}
                            showFavoritesLink={!showTabBar || !tabBarEnabled}
                            searchFormProps={searchFormProps}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'portalHeader',
                            })}
                        />
                    )}

                    <div className={cx('content', contentClassName)}>
                        {children}
                    </div>

                    {showFooter && (
                        <Footer
                            className={footerClassName}
                            wrapperClassName={footerWrapperClassName}
                            project={project}
                            footerType={footerType}
                            includeSupportBlock={footerIncludeSupportBlock}
                            isIndex={isIndex}
                            withoutSeparator={footerWithoutSeparator}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'portalFooter',
                            })}
                        />
                    )}

                    {showTabBar && tabBarEnabled && deviceType.isMobile && (
                        <TabBar />
                    )}
                </div>
            )}
        </DeviceTypeContext.Consumer>
    );
};

export default LayoutDefault;
