import React, {FunctionComponent, ReactElement} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {deviceMods} from 'utilities/stylesUtils';

import cx from './TwoColumnLayout.scss';

export interface IColumnProps extends IWithClassName {}

const Column: FunctionComponent<IColumnProps> = () => {
    return null;
};

interface ILeftColumnProps extends IWithClassName, IWithDeviceType {}

const LeftColumn: FunctionComponent<ILeftColumnProps> = ({
    className,
    children,
}) => {
    return <div className={cx('leftColumn', className)}>{children}</div>;
};

interface IRightColumnProps extends IWithClassName, IWithDeviceType {
    columnOffset: number;
    columnWidth: number;
    columnVisible: boolean;
    preserveColumnsOnMobile: boolean;
}

const RightColumn: FunctionComponent<IRightColumnProps> = ({
    className,
    children,
    columnOffset,
    columnWidth,
    columnVisible,
    deviceType,
    preserveColumnsOnMobile,
}) => {
    const styles =
        deviceType.isDesktop || preserveColumnsOnMobile
            ? {
                  marginLeft: `${columnOffset * 4}px`,
                  flexBasis: `${columnWidth * 4}px`,
              }
            : undefined;

    return columnVisible ? (
        <div className={cx('rightColumn', className)} style={styles}>
            {children}
        </div>
    ) : null;
};

export interface ITwoColumnLayoutProps extends IWithClassName, IWithDeviceType {
    tagName?: string;
    children: [ReactElement<IColumnProps>, ReactElement<IColumnProps>];
    rightColumnOffset: number;
    rightColumnWidth: number;
    rightColumnVisible?: boolean;
    preserveColumnsOnMobile?: boolean;
}

export const TwoColumnLayout: FunctionComponent<ITwoColumnLayoutProps> & {
    LeftColumn: typeof Column;
    RightColumn: typeof Column;
} = ({
    className,
    deviceType,
    children,
    tagName = 'div',
    rightColumnOffset,
    rightColumnWidth,
    rightColumnVisible = true,
    preserveColumnsOnMobile = false,
}) => {
    return React.createElement(
        tagName,
        {
            className: cx(
                'layout',
                deviceMods('layout', deviceType),
                {layout_preserveColumns: preserveColumnsOnMobile},
                className,
            ),
        },
        React.Children.map(
            children,
            (child: ReactElement<IColumnProps>, index) => {
                if (index === 0) {
                    return (
                        typeof child !== 'undefined' && (
                            <LeftColumn
                                key="left-column"
                                {...child.props}
                                deviceType={deviceType}
                            />
                        )
                    );
                } else if (index === 1) {
                    return (
                        typeof child !== 'undefined' && (
                            <RightColumn
                                key="right-column"
                                {...child.props}
                                preserveColumnsOnMobile={
                                    preserveColumnsOnMobile
                                }
                                columnVisible={rightColumnVisible}
                                columnOffset={rightColumnOffset}
                                columnWidth={rightColumnWidth}
                                deviceType={deviceType}
                            />
                        )
                    );
                }

                return null;
            },
        ),
    );
};

TwoColumnLayout.LeftColumn = Column;
TwoColumnLayout.RightColumn = Column;

export default TwoColumnLayout;
