import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import isLinkBreadcrumbsItem from './utilities/isLinkBreadcrumbsItem';
import getUrlFromProps from 'components/LinkBreadcrumbs/utilities/getUrlFromProps';
import getNameFromProps from 'components/LinkBreadcrumbs/utilities/getNameFromProps';

import LinkBreadcrumbsItem from './components/LinkBreadcrumbsItem/LinkBreadcrumbsItem';
import LinkBreadcrumbsArrow from './components/LinkBreadcrumbsArrow/LinkBreadcrumbsArrow';
import HorizontalHiddenScroll from 'components/HorizontalHiddenScroll/HorizontalHiddenScroll';
import SchemaMarkup, {
    getMetaBreadcrumbList,
} from 'components/SchemaMarkup/SchemaMarkup';

import cx from './LinkBreadcrumbs.scss';

interface IBreadcrumbsProps extends IWithQaAttributes, IWithClassName {
    scrollableClassName?: string;
    /**
     * Дочерние элементы должны быть обёрнуты `<Breadcrumbs.Item />`.
     * Тогда они будут разделяться стрелочками
     */
    children: React.ReactNode;

    /**
     * Включает поисковую разметку и передается в SchemaMarkup
     */
    nonce?: string;
}

function LinkBreadcrumbs(props: IBreadcrumbsProps): React.ReactElement | null {
    const {className, scrollableClassName, children, nonce} = props;

    const items = useMemo(() => {
        return React.Children.map(children, (child, index) => {
            if (!isLinkBreadcrumbsItem(child)) {
                return {
                    element: (
                        <React.Fragment key={index}>{child}</React.Fragment>
                    ),
                };
            }

            const element = (
                <React.Fragment key={child.key ?? index}>
                    {index !== 0 && <LinkBreadcrumbsArrow />}
                    {child}
                </React.Fragment>
            );

            if (!nonce) {
                return {element};
            }

            return {
                element,
                url: getUrlFromProps(child.props),
                name: getNameFromProps(child.props),
            };
        });
    }, [children, nonce]);

    if (!items || !items.length) {
        return null;
    }

    return (
        <>
            <HorizontalHiddenScroll
                className={cx(className)}
                {...prepareQaAttributes(props)}
                scrollableClassName={cx('root', scrollableClassName)}
            >
                {items.map(item => item.element)}
            </HorizontalHiddenScroll>

            {nonce && (
                <SchemaMarkup
                    data={getMetaBreadcrumbList(
                        items.map(({url, name}) => ({url, name})),
                    )}
                    nonce={nonce}
                />
            )}
        </>
    );
}

LinkBreadcrumbs.Item = LinkBreadcrumbsItem;

export default LinkBreadcrumbs;
