import React, {FC, MouseEvent, useContext} from 'react';

import {TMenuItemValue} from 'components/Menu/types';

import {
    IWithDataQaAttribute,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';

import {MenuContext} from 'components/Menu/Provider';
import IncludeIcon from 'icons/16/Include';

import cx from './MenuItem.scss';

export interface IMenuItemProps extends IWithDataQaAttribute {
    value: TMenuItemValue;
    selected?: boolean;
    disabled?: boolean;
    className?: string;
    enableSelectedIcon?: boolean;

    onClick?(e: MouseEvent): void;
}

const MenuItem: FC<IMenuItemProps> = ({
    value,
    children,
    selected,
    disabled,
    className,
    enableSelectedIcon,

    onClick,
    ...props
}) => {
    const {focused, select, focus, parentQa} = useContext(MenuContext);
    const handleClick = useImmutableCallback((e: React.MouseEvent) => {
        select(value);
        onClick?.(e);
    });
    const handleHover = useImmutableCallback(() => focus(value));

    return (
        <button
            type="button"
            className={cx('item', className)}
            role="option"
            tabIndex={-1}
            aria-disabled={disabled}
            aria-selected={selected}
            data-focused={focused === value}
            disabled={disabled}
            onMouseOver={disabled ? undefined : handleHover}
            onClick={handleClick}
            {...prepareQaAttributes({
                key: value,
                parent: parentQa,
                current: 'option',
            })}
            {...props}
        >
            {enableSelectedIcon && (
                <div className={cx('selectedIcon')}>
                    {selected && <IncludeIcon />}
                </div>
            )}
            {children}
        </button>
    );
};

export default MenuItem;
