import React, {useEffect} from 'react';

import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {withZIndex} from '@yandex-lego/components/withZIndex';
import {MessageBoxPopup as MessageBoxPopupDesktop} from '@yandex-lego/components/MessageBox/desktop/bundle';

import LayersContext from 'contexts/LayersContext';

import './MessageBoxPopup.scss';

export enum EMessageBoxPopupTheme {
    DEFAULT = 'default',
    WHITE = 'white',
}

export interface IMessageBoxPopupProps extends IWithClassName {
    isVisible: boolean;
    /**
     * Элемент, относительно которого позиционируется Tooltip
     **/
    anchorRef?: React.RefObject<HTMLElement>;
    /**
     * Ссылка на DOM-элемент, в котором размещается попап
     * Важно, чтобы контейнер имел position: relative для корректного позициоинирования.
     **/
    scopeRef?: React.RefObject<HTMLElement>;
    /**
     * Ссылка на DOM-элемент, относительно которого будет расчитана границы отображения попапа.
     **/
    boundaryRef?: React.RefObject<HTMLElement>;
    /**
     * Откуда из анкора растет хвост
     **/
    direction?: EPopupDirection | EPopupDirection[];
    /**
     * Отступ хвостика вдоль поверхности попапа
     * в данный момент не работает https://st.yandex-team.ru/LEGOSUPPORT-390
     **/
    tailOffset?: number;
    /**
     * Отступ всего MessageBoxPopup вдоль основного направления (вдоль хвостика)
     **/
    mainOffset?: number;
    /**
     * Отступ всего MessageBoxPopup вдоль побочного направления (перпендикулярно хвостику)
     * сейчас используется как костыль чтобы выровнять хвостик https://st.yandex-team.ru/LEGOSUPPORT-390
     **/
    secondaryOffset?: number;
    /**
     * Минимальный отступ от границ вьюпорта
     **/
    viewportOffset?: number;

    hasClose?: boolean;

    zIndexGroupLevel?: number;
    onClose?: (
        e: React.MouseEvent<HTMLButtonElement, MouseEvent> | Event,
    ) => void;

    theme?: EMessageBoxPopupTheme;

    closeOnScroll?: boolean;

    children?: React.ReactNode;
}

const DIRECTION_DEFAULT = [
    EPopupDirection.TOP,
    EPopupDirection.BOTTOM,
    EPopupDirection.TOP_LEFT,
    EPopupDirection.TOP_RIGHT,
    EPopupDirection.BOTTOM_LEFT,
    EPopupDirection.BOTTOM_RIGHT,
];

const LegoMessageBoxPopup = withZIndex(MessageBoxPopupDesktop);

/**
 * Общий компонент "тултипа" (всплывашка с хвостиком).
 * Название выбрано для корреляции с @yandex-lego/components/MessageBoxPopup
 **/
const MessageBoxPopup: React.FC<IMessageBoxPopupProps> = props => {
    const {
        anchorRef,
        direction = DIRECTION_DEFAULT,
        className,
        children,
        closeOnScroll,
        onClose,
        hasClose,
        isVisible,
        scopeRef,
        tailOffset,
        mainOffset,
        boundaryRef,
        viewportOffset = 4,
        secondaryOffset,
        zIndexGroupLevel,
        theme = EMessageBoxPopupTheme.DEFAULT,
    } = props;

    useEffect(() => {
        if (!closeOnScroll || !onClose) {
            return;
        }

        window.addEventListener('scroll', onClose);

        return () => {
            window.removeEventListener('scroll', onClose);
        };
    }, [closeOnScroll, onClose]);

    return (
        <LayersContext.Consumer>
            {(layer): React.ReactNode => (
                <LegoMessageBoxPopup
                    // Пропсы, которые будут прокинуты в Лего Popup
                    zIndexGroupLevel={zIndexGroupLevel || layer}
                    // ToDo: костыль, убрать когда починят лего https://st.yandex-team.ru/ISL-9648
                    zIndex={(zIndexGroupLevel || layer) * 1000}
                    visible={isVisible}
                    anchor={anchorRef}
                    direction={direction}
                    scope={scopeRef}
                    tailOffset={tailOffset}
                    mainOffset={mainOffset}
                    viewportOffset={viewportOffset}
                    secondaryOffset={secondaryOffset}
                    // Остальные пропсы прокидываются в Лего MessageBox
                    hasTail={true}
                    view={
                        theme === EMessageBoxPopupTheme.DEFAULT
                            ? 'default'
                            : 'inverse'
                    }
                    opaque={theme === EMessageBoxPopupTheme.WHITE}
                    boundary={boundaryRef}
                    size="s"
                    onClose={onClose}
                    className={className}
                    hasClose={hasClose}
                >
                    {children}
                </LegoMessageBoxPopup>
            )}
        </LayersContext.Consumer>
    );
};

export default MessageBoxPopup;
